/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import org.mule.soap.api.client.BadRequestException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.internal.util.XmlTransformationException;
import org.mule.soap.internal.util.XmlTransformationUtils;
import org.mule.soap.internal.xml.util.XMLUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.util.Map;
import java.util.Optional;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.mule.soap.internal.xml.util.XMLUtils.getElements;

/**
 * Abstract implementation for a request enricher that adds a node for each sent attachment to the incoming SOAP
 * request with all the information required to send the attachments using the SOAP protocol.
 *
 * @since 1.0
 */
public abstract class AttachmentRequestEnricher {

  private static final String TEXT_XPATH_MASK = "//*[normalize-space()='cid:%s']/text()";

  /**
   * @param body        the XML SOAP body provided by the user.
   * @param attachments the attachments to upload.
   */
  public String enrichRequest(String body, Map<String, SoapAttachment> attachments) {
    try {
      Document bodyDocument = XmlTransformationUtils.stringToDocument(body);
      Element documentElement = bodyDocument.getDocumentElement();
      attachments.forEach((name, attachment) -> {
        Optional<Node> placeholderNode = getAttachmentPlaceholderNode(bodyDocument, name);
        Node attachmentNode = createAttachmentNode(bodyDocument, name, attachment);
        if (placeholderNode.isPresent()) {
          Node placeholder = placeholderNode.get();
          placeholder.getParentNode().replaceChild(attachmentNode, placeholder);
        } else {
          Element attachmentElement = bodyDocument.createElement(name);
          attachmentElement.appendChild(attachmentNode);
          documentElement.appendChild(attachmentElement);
        }
      });
      return XMLUtils.toXml(bodyDocument);
    } catch (XmlTransformationException e) {
      throw new BadRequestException("Error while preparing request for the provided body", e);
    }
  }

  protected Optional<Node> getAttachmentPlaceholderNode(Document bodyDocument, String name) {
    NodeList textNodes = getElements(format(TEXT_XPATH_MASK, name), bodyDocument);
    return ofNullable(textNodes.getLength() != 0 ? textNodes.item(0) : null);
  }

  /**
   * Creates the content of the attachment node to be added in the SOAP request
   *
   * @param bodyDocument      the document where we are adding the node element.
   * @param attachment      the attachment to be sent.
   *
   * @return the attachment node
   */
  abstract Node createAttachmentNode(Document bodyDocument, String name, SoapAttachment attachment);

}
