/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import com.google.common.collect.ImmutableMap;
import org.apache.cxf.message.Exchange;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.soap.api.exception.EncodingException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.internal.client.AbstractSoapCxfClient;
import org.mule.wsdl.parser.model.operation.OperationModel;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Base64;
import java.util.Collection;
import java.util.List;

import static java.lang.String.format;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getLocalPart;

/**
 * {@link AttachmentResponseEnricher} implementation for SOAP with attachments.
 *
 * @since 1.0
 */
public final class SoapAttachmentResponseEnricher extends AttachmentResponseEnricher {


  public SoapAttachmentResponseEnricher(List<OperationModel> operations) {
    super(operations);
  }

  /**
   * {@inheritDoc}
   * <p>
   * Extracts the base64 encoded content from the attachment nodes, decodes them and then remove all the nodes to
   * clean the response body.
   */
  @Override
  protected void processResponseAttachments(Document response, Collection<ObjectFieldType> attachments, Exchange exchange) {
    ImmutableMap.Builder<String, SoapAttachment> builder = ImmutableMap.builder();
    attachments.forEach(attachment -> {
      String attachmentName = getLocalPart(attachment);
      builder.put(attachmentName, getAttachment(response, attachmentName));
    });
    exchange.put(AbstractSoapCxfClient.MULE_ATTACHMENTS_KEY, builder.build());
  }

  /**
   * Extracts the base64 encoded content from the attachment {@code name}, decodes its content and removes the node from the
   * document.
   */
  private SoapAttachment getAttachment(Document response, String name) {
    Node attachmentNode = response.getDocumentElement().getElementsByTagNameNS("*", name).item(0);
    InputStream decodedAttachment = decodeAttachment(name, attachmentNode.getTextContent());
    response.getDocumentElement().removeChild(attachmentNode);
    return new SoapAttachment(decodedAttachment, "*/*");
  }

  /**
   * Decodes the attachment content from base64.
   */
  private InputStream decodeAttachment(String name, String attachmentContent) {
    try {
      return new ByteArrayInputStream(Base64.getDecoder().decode(attachmentContent));
    } catch (Exception e) {
      throw new EncodingException(format("Cannot decode base64 attachment [%s]", name));
    }
  }
}
