/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api;


import org.mule.soap.api.security.SecurityStrategy;
import org.mule.soap.api.transport.locator.DefaultTransportResourceLocator;
import org.mule.soap.api.transport.locator.TransportResourceLocator;

import java.util.ArrayList;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.soap.api.SoapVersion.SOAP_11;

/**
 * Builder pattern implementation that creates {@link SoapWebServiceConfiguration} instances.
 *
 * @since 1.0
 */
public class SoapWebServiceConfigurationBuilder {

  private String wsdlLocation;
  private String address;
  private String service;
  private String port;
  private SoapVersion version = SOAP_11;
  private boolean isMtom;
  private List<SecurityStrategy> securities = new ArrayList<>();
  private String encoding;
  private TransportResourceLocator locator = new DefaultTransportResourceLocator();

  SoapWebServiceConfigurationBuilder() {}

  /**
   * Sets the location of the WSDL that describes the SOAP web service. This can be an URL to an
   * external resource (e.g http://somewsdl.com/hit?wsdl), just a reference to a local file or an application resource name.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withWsdlLocation(String wsdlLocation) {
    this.wsdlLocation = wsdlLocation;
    return this;
  }

  /**
   * Sets the address of the Web Service, if none specified it will be fetched from the wsdl file, if possible.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withAddress(String address) {
    this.address = address;
    return this;
  }

  /**
   * Sets the service of the WSDL we want to perform operations from.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withService(String service) {
    this.service = service;
    return this;
  }

  /**
   * Sets the port of the service that describes the set of operations that can be performed.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withPort(String port) {
    this.port = port;
    return this;
  }

  /**
   * Sets the encoding of the messages send and retrieved by the .
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withEncoding(String encoding) {
    this.encoding = encoding;
    return this;
  }

  /**
   * Sets the {@link SoapVersion} of the Web Service. defaults to SOAP 1.1
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withVersion(SoapVersion version) {
    this.version = version;
    return this;
  }

  /**
   * Specifies that the Web Service is MTOM enabled.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder enableMtom(boolean mtomEnabled) {
    this.isMtom = mtomEnabled;
    return this;
  }

  /**
   * Sets a new {@link SecurityStrategy} to connect with a Secured Soap Web Service.
   * <p>
   * Multiple {@link SecurityStrategy Security Strategies} can be configured.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withSecurity(SecurityStrategy security) {
    this.securities.add(security);
    return this;
  }

  /**
   * Sets a list of {@link SecurityStrategy SecurityStrategies} to connect with a Secured Soap Web Service.
   *
   * @return this builder.
   */
  public SoapWebServiceConfigurationBuilder withSecurities(List<SecurityStrategy> security) {
    this.securities.addAll(security);
    return this;
  }

  /**
   * Sets a {@link TransportResourceLocator} instance to fetch the wsdl resources.
   *
   * @return this builder
   */
  public SoapWebServiceConfigurationBuilder withResourceLocator(TransportResourceLocator locator) {
    this.locator = locator;
    return this;
  }

  /**
   * @return a new {@link SoapWebServiceConfiguration} instance with the attributes specified.
   */
  public SoapWebServiceConfiguration build() {
    checkNotNull(wsdlLocation, "WSDL location cannot be null");
    checkNotNull(service, "Service cannot be null");
    checkNotNull(port, "Port cannot be null");
    return new SoapWebServiceConfiguration(wsdlLocation, address, service, port, version, isMtom, securities, locator, encoding);
  }
}
