/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import org.mule.soap.api.exception.EncodingException;
import org.mule.soap.api.message.SoapAttachment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.io.InputStream;
import java.util.Base64;

import static org.apache.commons.io.IOUtils.toByteArray;

/**
 * {@link AttachmentRequestEnricher} implementation to send attachments using SOAP with Attachments.
 *
 * @since 1.0
 */
public final class SoapAttachmentRequestEnricher extends AttachmentRequestEnricher {

  /**
   * {@inheritDoc}
   * <p>
   * Creates the attachment content encoded to Base64 plain in the XML Request in the generated attachment node.
   */
  @Override
  protected Node createAttachmentNode(Document bodyDocument, String name, SoapAttachment attachment) {
    // Encode the attachment to base64 to be sent as SOAP with Attachments.
    return bodyDocument.createTextNode(toBase64(attachment.getContent()));
  }

  /**
   * Returns an {@link String} with the content of the attachment encoded to Base64.
   *
   * @param content the content of the attachment.
   */
  private String toBase64(InputStream content) {
    try {
      return Base64.getEncoder().encodeToString(toByteArray(content));
    } catch (Exception e) {
      throw new EncodingException("Could not encode attachment content to base64", e);
    }
  }
}
