/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import static java.lang.String.format;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_ATTACHMENTS_KEY;
import org.mule.soap.api.exception.EncodingException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.wsdl.parser.model.operation.OperationModel;

import com.google.common.collect.ImmutableMap;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Base64;
import java.util.List;
import java.util.Map;

import org.apache.cxf.message.Exchange;

/**
 * {@link AttachmentResponseEnricher} implementation for SOAP with attachments.
 *
 * @since 1.0
 */
public final class SoapAttachmentResponseEnricher extends AttachmentResponseEnricher {


  public SoapAttachmentResponseEnricher(List<OperationModel> operations) {
    super(operations);
  }

  @Override
  protected void processAttachmentData(String attachmentName, String attachmentData, Exchange exchange) {
    ImmutableMap.Builder<String, SoapAttachment> builder = ImmutableMap.builder();

    if (exchange.get(MULE_ATTACHMENTS_KEY) != null) {
      builder.putAll((Map<String, SoapAttachment>) exchange.get(MULE_ATTACHMENTS_KEY));
    }

    builder.put(attachmentName, buildAttachment(attachmentName, attachmentData));

    exchange.put(MULE_ATTACHMENTS_KEY, builder.build());
  }

  private SoapAttachment buildAttachment(String attachmentName, String attachmentData) {
    InputStream decodedAttachment = decodeAttachment(attachmentName, attachmentData);
    return new SoapAttachment(decodedAttachment, "*/*");
  }

  /**
   * Decodes the attachment content from base64.
   */
  private InputStream decodeAttachment(String name, String attachmentContent) {
    try {
      return new ByteArrayInputStream(Base64.getDecoder().decode(attachmentContent));
    } catch (Exception e) {
      throw new EncodingException(format("Cannot decode base64 attachment [%s]", name));
    }
  }
}
