/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security;

import static java.util.Optional.empty;
import static org.apache.wss4j.common.ConfigurationConstants.TIMESTAMP;
import static org.apache.wss4j.common.ConfigurationConstants.TTL_FUTURE_TIMESTAMP;

import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableMap;

/**
 * Verifies the timestamp of an incoming SOAP message.
 * <p>
 * This should be used when an incoming soap message is encrypted, signed, etc, and also includes a timestamp that express
 * the creation and expiration of the security semantic of the message.
 *
 * @see <a href="http://docs.oasis-open.org/wss/v1.1/wss-v1.1-spec-errata-os-SOAPMessageSecurity.htm#_Toc118717167">Security Timestamps</a>
 *
 * @since 1.2.0
 */
public class WssIncomingTimestampSecurityStrategy implements SecurityStrategy {

  /**
   * Specifies the time in seconds in the future within which the Created time of an incoming Timestamp is valid. The
   * default value is "60", to avoid problems where clocks are slightly askew. To reject all future-created Timestamps,
   * set this value to "0".
   */
  private long futureTimeToLive;

  public WssIncomingTimestampSecurityStrategy(long futureTimeToLive) {
    this.futureTimeToLive = futureTimeToLive;
  }

  @Override
  public String securityAction() {
    return TIMESTAMP;
  }

  @Override
  public SecurityStrategyType securityType() {
    return SecurityStrategyType.INCOMING;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return empty();
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    return ImmutableMap.<String, Object>builder().put(TTL_FUTURE_TIMESTAMP, String.valueOf(futureTimeToLive)).build();
  }

}
