/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security;

import static java.util.Optional.of;
import static org.apache.wss4j.common.ConfigurationConstants.SIGNATURE_USER;
import static org.apache.wss4j.common.ConfigurationConstants.SIG_PROP_REF_ID;
import static org.apache.wss4j.common.ext.WSPasswordCallback.SIGNATURE;

import org.mule.soap.api.security.configuration.WssSignConfiguration;
import org.mule.soap.api.security.stores.WssKeyStoreConfiguration;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableMap;
import org.apache.wss4j.dom.handler.WSHandlerConstants;

/**
 * Signs the SOAP request that is being sent, using the private key of the key-store in the provided TLS context.
 *
 * @since 1.0
 */
public class WssSignSecurityStrategy implements SecurityStrategy {

  private static final String WS_SIGN_PROPERTIES_KEY = "signProperties";

  /**
   * The keystore to use when signing the message.
   */
  private WssKeyStoreConfiguration keyStoreConfiguration;

  /**
   * Signing configuration.
   */
  private WssSignConfiguration wssSignConfiguration;

  public WssSignSecurityStrategy(WssKeyStoreConfiguration keyStoreConfiguration) {
    this.keyStoreConfiguration = keyStoreConfiguration;
  }

  public WssSignSecurityStrategy(WssKeyStoreConfiguration keyStoreConfiguration, WssSignConfiguration wssSignConfiguration) {
    this.keyStoreConfiguration = keyStoreConfiguration;
    this.wssSignConfiguration = wssSignConfiguration;
  }

  @Override
  public SecurityStrategyType securityType() {
    return SecurityStrategyType.OUTGOING;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return of(new WSPasswordCallbackHandler(SIGNATURE, cb -> cb.setPassword(keyStoreConfiguration.getPassword())));
  }

  @Override
  public String securityAction() {
    return WSHandlerConstants.SIGNATURE;
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    ImmutableMap.Builder<String, Object> stringObjectBuilder = ImmutableMap.<String, Object>builder()
        .put(SIG_PROP_REF_ID, WS_SIGN_PROPERTIES_KEY)
        .put(WS_SIGN_PROPERTIES_KEY, keyStoreConfiguration.getConfigurationProperties())
        .put(SIGNATURE_USER, keyStoreConfiguration.getAlias());

    if (wssSignConfiguration != null) {
      stringObjectBuilder.putAll((Map) wssSignConfiguration.getConfigurationProperties());
    }

    return stringObjectBuilder.build();
  }
}
