/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security.configuration;

import static org.apache.wss4j.common.ConfigurationConstants.ALLOW_RSA15_KEY_TRANSPORT_ALGORITHM;
import static org.apache.wss4j.common.ConfigurationConstants.ENCRYPTION_PARTS;
import static org.apache.wss4j.common.ConfigurationConstants.ENC_DIGEST_ALGO;
import static org.apache.wss4j.common.ConfigurationConstants.ENC_KEY_ID;
import static org.apache.wss4j.common.ConfigurationConstants.ENC_KEY_TRANSPORT;
import static org.apache.wss4j.common.ConfigurationConstants.ENC_SYM_ALGO;
import static org.apache.wss4j.common.WSS4JConstants.KEYTRANSPORT_RSA15;

import java.util.List;
import java.util.Properties;

/**
 * Security configuration regarding encryption algorithm.
 *
 * @since 1.1
 */
public class WssEncryptionConfiguration {

  private String encryptionKeyIdentifier;
  private String encryptionSymAlgorithm;
  private String encryptionKeyTransportAlgorithm;
  private String encryptionDigestAlgorithm;
  private List<WssPart> wssParts;

  public WssEncryptionConfiguration(String encryptionKeyIdentifier,
                                    String encryptionSymAlgorithm,
                                    String encryptionKeyTransportAlgorithm,
                                    String encryptionDigestAlgorithm,
                                    List<WssPart> wssParts) {
    this.encryptionKeyIdentifier = encryptionKeyIdentifier;
    this.encryptionSymAlgorithm = encryptionSymAlgorithm;
    this.encryptionKeyTransportAlgorithm = encryptionKeyTransportAlgorithm;
    this.encryptionDigestAlgorithm = encryptionDigestAlgorithm;
    this.wssParts = wssParts;
  }

  public String getEncryptionKeyIdentifier() {
    return encryptionKeyIdentifier;
  }

  public String getEncryptionSymAlgorithm() {
    return encryptionSymAlgorithm;
  }

  public String getEncryptionKeyTransportAlgorithm() {
    return encryptionKeyTransportAlgorithm;
  }

  public String getEncryptionDigestAlgorithm() {
    return encryptionDigestAlgorithm;
  }

  public List<WssPart> getWssParts() {
    return wssParts;
  }

  public Properties getConfigurationProperties() {
    Properties properties = new Properties();

    if (encryptionKeyIdentifier != null) {
      properties.setProperty(ENC_KEY_ID, encryptionKeyIdentifier);
    }
    if (encryptionSymAlgorithm != null) {
      properties.setProperty(ENC_SYM_ALGO, encryptionSymAlgorithm);
    }
    if (encryptionKeyTransportAlgorithm != null) {
      if (encryptionKeyTransportAlgorithm.equals(KEYTRANSPORT_RSA15)) {
        properties.setProperty(ALLOW_RSA15_KEY_TRANSPORT_ALGORITHM, "true");
      }
      properties.setProperty(ENC_KEY_TRANSPORT, encryptionKeyTransportAlgorithm);
    }
    if (encryptionDigestAlgorithm != null) {
      properties.setProperty(ENC_DIGEST_ALGO, encryptionDigestAlgorithm);
    }
    if (wssParts != null) {
      StringBuilder sb = new StringBuilder();
      for (WssPart wssPart : wssParts) {
        sb.append(wssPart.toString());
      }
      properties.setProperty(ENCRYPTION_PARTS, sb.toString());
    }

    return properties;
  }

}
