/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.client;

import static java.util.Collections.emptyList;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.cxf.message.Message.ENCODING;
import static org.mule.soap.internal.util.XmlTransformationUtils.stringToDomElement;

import org.mule.soap.api.client.BadRequestException;
import org.mule.soap.api.client.SoapClient;
import org.mule.soap.api.exception.SoapFaultException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.api.message.SoapRequest;
import org.mule.soap.api.message.SoapResponse;
import org.mule.soap.api.transport.DispatcherException;
import org.mule.soap.api.transport.TransportDispatcher;
import org.mule.soap.internal.generator.SoapRequestGenerator;
import org.mule.soap.internal.generator.SoapResponseGenerator;
import org.mule.soap.internal.generator.attachment.AttachmentRequestEnricher;
import org.mule.soap.internal.generator.attachment.AttachmentResponseEnricher;
import org.mule.wsdl.parser.exception.OperationNotFoundException;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.WsdlModel;
import org.mule.wsdl.parser.model.operation.OperationModel;
import org.mule.wsdl.parser.model.operation.OperationType;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamReader;

import org.apache.cxf.binding.soap.SoapFault;
import org.apache.cxf.binding.soap.SoapHeader;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.endpoint.Endpoint;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Attachment;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.ExchangeImpl;
import org.apache.cxf.service.model.BindingOperationInfo;

/**
 * Base {@link SoapClient} implementation for clients that works on top of CXF.
 *
 * @since 1.0
 */
public abstract class AbstractSoapCxfClient implements SoapClient {

  public static final String MESSAGE_DISPATCHER = "mule.soap.dispatcher";
  public static final String MULE_ATTACHMENTS_KEY = "mule.soap.attachments";
  public static final String MULE_WSC_ADDRESS = "mule.soap.address";
  public static final String MULE_HEADERS_KEY = "mule.soap.headers";
  public static final String MULE_TRANSPORT_HEADERS_KEY = "mule.soap.transport.headers";
  public static final String MULE_TRANSPORT_ADDITIONAL_DATA_KEY = "mule.soap.transport.additionalData";
  public static final String MULE_SOAP_ACTION = "mule.soap.action";
  public static final String MULE_SOAP_OPERATION_STYLE = "mule.soap.operation.type";

  private final SoapRequestGenerator requestGenerator;
  private final SoapResponseGenerator responseGenerator;

  private final Client client;
  private final WsdlModel wsdlModel;
  private final PortModel port;
  private final String address;
  private final String encoding;

  AbstractSoapCxfClient(Client client,
                        WsdlModel wsdlModel,
                        PortModel portModel,
                        String address,
                        String encoding,
                        AttachmentRequestEnricher requestEnricher,
                        AttachmentResponseEnricher responseEnricher) {
    this.client = client;
    this.wsdlModel = wsdlModel;
    this.port = portModel;
    this.address = address;
    this.encoding = encoding;
    // TODO: MULE-10889 -> instead of creating this enrichers, interceptors that works with the live stream would be ideal
    this.requestGenerator = new SoapRequestGenerator(requestEnricher, portModel);
    this.responseGenerator = new SoapResponseGenerator(responseEnricher);
  }

  @Override
  public SoapResponse consume(SoapRequest request, TransportDispatcher dispatcher) {
    requireNonNull(dispatcher, "Message Dispatcher cannot be null");
    String operation = request.getOperation();
    Exchange exchange = new ExchangeImpl();
    Object[] response = invoke(request, exchange, dispatcher);
    return responseGenerator.generate(operation, response, exchange);
  }

  @Override
  public void destroy() {
    client.destroy();
  }

  private Object[] invoke(SoapRequest request, Exchange exchange, TransportDispatcher dispatcher) {
    String ope = request.getOperation();
    XMLStreamReader xmlBody = getXmlBody(request);
    try {
      Map<String, Object> ctx = getInvocationContext(request, dispatcher);
      return client.invoke(getInvocationOperation(ope), new Object[] {xmlBody}, ctx, exchange);
    } catch (SoapFault sf) {
      throw new SoapFaultException(sf);
    } catch (Fault f) {
      if (f.getMessage().contains("COULD_NOT_READ_XML")) {
        throw new BadRequestException("Error consuming the operation [" + ope + "], the request body is not a valid XML");
      }
      throw new SoapFaultException(f);
    } catch (OperationNotFoundException e) {
      String errorMsg = "The provided operation [" + ope + "] does not exist in the WSDL file [" + wsdlModel.getLocation() + "]";
      throw new BadRequestException(errorMsg, e);
    } catch (DispatcherException e) {
      throw e;
    } catch (Exception e) {
      throw new RuntimeException("Unexpected error while consuming the web service operation [" + ope + "]", e);
    }
  }

  private XMLStreamReader getXmlBody(SoapRequest request) {
    return requestGenerator.generate(request.getOperation(), request.getContent(), getEncoding(request),
                                     request.getAttachments());
  }

  private BindingOperationInfo getInvocationOperation(String operationName) {
    // Normally its not this hard to invoke the CXF Client, but we're
    // sending along some exchange properties, so we need to use a more advanced
    // method
    Endpoint ep = client.getEndpoint();
    // The operation is always named invoke because hits our ProxyService implementation.
    String method = port.getOperation(operationName).getType().equals(OperationType.ONE_WAY) ? "invokeOneWay" : "invoke";
    QName q = new QName(ep.getService().getName().getNamespaceURI(), method);
    BindingOperationInfo bop = ep.getBinding().getBindingInfo().getOperation(q);
    if (bop.isUnwrappedCapable()) {
      bop = bop.getUnwrappedOperation();
    }
    return bop;
  }

  private Map<String, Object> getInvocationContext(SoapRequest request, TransportDispatcher dispatcher) {
    Map<String, Object> props = new HashMap<>();
    OperationModel operation = port.getOperation(request.getOperation());

    // is NOT mtom the attachments must not be touched by cxf, we create a custom request embedding the attachment in the xml
    props.put(MULE_ATTACHMENTS_KEY, buildCxfAttachments(request.getAttachments()));
    props.put(MULE_WSC_ADDRESS, address);
    props.put(ENCODING, getEncoding(request));
    props.put(MULE_HEADERS_KEY, buildCxfHeaders(request.getSoapHeaders()));
    props.put(MULE_TRANSPORT_HEADERS_KEY, request.getTransportHeaders());
    props.put(MESSAGE_DISPATCHER, dispatcher);
    props.put(MULE_SOAP_OPERATION_STYLE, port.getOperation(request.getOperation()).getType());
    String soapAction = operation.getSoapAction();
    if (soapAction != null) {
      props.put(MULE_SOAP_ACTION, soapAction);
    }
    Map<String, Object> ctx = new HashMap<>();
    ctx.put(Client.REQUEST_CONTEXT, props);
    return ctx;
  }

  private String getEncoding(SoapRequest request) {
    try {
      return new ContentType(request.getContentType()).getParameter("charset");
    } catch (ParseException e) {
      return isBlank(encoding) ? "UTF-8" : encoding;
    }
  }

  protected abstract Map<String, Attachment> buildCxfAttachments(Map<String, SoapAttachment> attachments);

  private List<SoapHeader> buildCxfHeaders(Map<String, String> headers) {
    if (headers == null) {
      return emptyList();
    }
    return headers.entrySet().stream()
        .map(header -> {
          try {
            return new SoapHeader(new QName(null, header.getKey()), stringToDomElement(header.getValue()));
          } catch (Exception e) {
            throw new BadRequestException("Cannot parse input header [" + header.getKey() + "]", e);
          }
        })
        .collect(toList());
  }
}
