/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.message;

import com.google.common.collect.ImmutableMap;
import org.mule.soap.internal.message.DefaultSoapRequest;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A builder implementation to create {@link SoapRequest}s
 *
 * @since 1.0
 */
public class SoapRequestBuilder {

  private InputStream content;
  private ImmutableMap.Builder<String, String> soapHeaders = ImmutableMap.builder();
  private ImmutableMap.Builder<String, String> transportHeaders = ImmutableMap.builder();
  private ImmutableMap.Builder<String, SoapAttachment> attachments = ImmutableMap.builder();
  private String contentType = "application/xml";
  private String operation;
  private boolean useXMLInitialDeclaration;

  SoapRequestBuilder() {}

  public SoapRequestBuilder content(InputStream content) {
    this.content = content;
    return this;
  }

  public SoapRequestBuilder content(String content) {
    this.content = new ByteArrayInputStream(content.getBytes());
    return this;
  }

  public SoapRequestBuilder soapHeaders(Map<String, String> soapHeaders) {
    this.soapHeaders.putAll(soapHeaders);
    return this;
  }

  public SoapRequestBuilder transportHeader(String key, String value) {
    this.transportHeaders.put(key, value);
    return this;
  }

  public SoapRequestBuilder transportHeaders(Map<String, String> headers) {
    this.transportHeaders.putAll(headers);
    return this;
  }

  public SoapRequestBuilder attachment(String name, SoapAttachment attachment) {
    this.attachments.put(name, attachment);
    return this;
  }

  public SoapRequestBuilder attachments(Map<String, SoapAttachment> attachments) {
    this.attachments.putAll(attachments);
    return this;
  }

  public SoapRequestBuilder contentType(String contentType) {
    this.contentType = contentType;
    return this;
  }

  public SoapRequestBuilder operation(String operation) {
    this.operation = operation;
    return this;
  }

  public SoapRequestBuilder useXMLInitialDeclaration(boolean useXMLInitialDeclaration) {
    this.useXMLInitialDeclaration = useXMLInitialDeclaration;
    return this;
  }

  public DefaultSoapRequest build() {
    checkNotNull(operation, "Missing executing operation");
    return new DefaultSoapRequest(content,
                                  soapHeaders.build(),
                                  transportHeaders.build(),
                                  attachments.build(),
                                  contentType,
                                  operation,
                                  useXMLInitialDeclaration);
  }
}
