/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.client;

import static java.lang.Boolean.TRUE;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.cxf.interceptor.StaxOutInterceptor.FORCE_START_DOCUMENT;
import static org.apache.cxf.interceptor.StaxInEndingInterceptor.STAX_IN_NOCLOSE;
import static org.apache.cxf.message.Message.CONTENT_TYPE;
import static org.apache.cxf.message.Message.ENCODING;
import static org.mule.soap.internal.util.XmlTransformationUtils.stringToDomElement;

import org.mule.soap.api.client.BadRequestException;
import org.mule.soap.api.client.BadResponseException;
import org.mule.soap.api.client.SoapClient;
import org.mule.soap.api.exception.SoapFaultException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.api.message.SoapRequest;
import org.mule.soap.api.message.SoapResponse;
import org.mule.soap.api.transport.DispatcherException;
import org.mule.soap.api.transport.TransportDispatcher;
import org.mule.soap.api.transport.TransportResponse;
import org.mule.soap.internal.generator.SoapRequestGenerator;
import org.mule.soap.internal.generator.SoapResponseGenerator;
import org.mule.soap.internal.generator.attachment.AttachmentRequestEnricher;
import org.mule.soap.internal.generator.attachment.AttachmentResponseEnricher;
import org.mule.soap.internal.util.OperationResolver;
import org.mule.wsdl.parser.exception.OperationNotFoundException;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.WsdlModel;
import org.mule.wsdl.parser.model.operation.OperationModel;
import org.mule.wsdl.parser.model.operation.OperationType;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamReader;

import com.google.common.net.MediaType;
import org.apache.cxf.binding.soap.SoapFault;
import org.apache.cxf.binding.soap.SoapHeader;
import org.apache.cxf.Bus;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.endpoint.ClientImpl;
import org.apache.cxf.endpoint.Endpoint;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Attachment;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.ExchangeImpl;
import org.apache.cxf.message.Message;
import org.apache.cxf.message.MessageImpl;
import org.apache.cxf.service.model.BindingOperationInfo;
import org.apache.cxf.service.Service;
import org.apache.cxf.transport.Conduit;

/**
 * Base {@link SoapClient} implementation for clients that works on top of CXF.
 *
 * @since 1.0
 */
public abstract class AbstractSoapCxfClient implements SoapClient {

  public static final String MESSAGE_DISPATCHER = "mule.soap.dispatcher";
  public static final String MULE_ATTACHMENTS_KEY = "mule.soap.attachments";
  public static final String MULE_WSC_ADDRESS = "mule.soap.address";
  public static final String MULE_HEADERS_KEY = "mule.soap.headers";
  public static final String MULE_TRANSPORT_HEADERS_KEY = "mule.soap.transport.headers";
  public static final String MULE_TRANSPORT_ADDITIONAL_DATA_KEY = "mule.soap.transport.additionalData";
  public static final String MULE_SOAP_ACTION = "mule.soap.action";
  public static final String MULE_SOAP_OPERATION_STYLE = "mule.soap.operation.type";

  private final SoapRequestGenerator requestGenerator;
  private final SoapResponseGenerator responseGenerator;

  private final Client client;
  private final WsdlModel wsdlModel;
  private final OperationResolver operationResolver;
  private final String address;
  private final String encoding;

  AbstractSoapCxfClient(Client client,
                        WsdlModel wsdlModel,
                        PortModel portModel,
                        String address,
                        String encoding,
                        AttachmentRequestEnricher requestEnricher,
                        AttachmentResponseEnricher responseEnricher) {
    this.client = client;
    this.wsdlModel = wsdlModel;
    this.operationResolver = new OperationResolver(portModel.getOperationsMap());
    this.address = address;
    this.encoding = encoding;
    // TODO: MULE-10889 -> instead of creating this enrichers, interceptors that works with the live stream would be ideal
    this.requestGenerator = new SoapRequestGenerator(requestEnricher, operationResolver);
    this.responseGenerator = new SoapResponseGenerator(responseEnricher);
  }

  @Override
  public SoapResponse consume(SoapRequest request, TransportDispatcher dispatcher) {
    requireNonNull(dispatcher, "Message Dispatcher cannot be null");
    String operation = request.getOperation();
    Exchange exchange = new ExchangeImpl();
    Object[] response = invoke(request, exchange, dispatcher);
    return responseGenerator.generate(operation, response, exchange);
  }

  @Override
  public SoapResponse parseResponse(String operation, TransportResponse transportResponse) {
    requireNonNull(transportResponse, "Transport response cannot be null");
    Exchange exchange = new ExchangeImpl();
    Object[] response = invoke(operation, transportResponse, exchange);
    return responseGenerator.generate(operation, response, exchange);
  }

  @Override
  public void destroy() {
    client.destroy();
  }

  public void addOperations(Map<String, OperationModel> operations) {
    requireNonNull(operations, "Operations cannot be null");
    operationResolver.add(operations);
  }

  private Object[] invoke(SoapRequest request, Exchange exchange, TransportDispatcher dispatcher) {
    String ope = request.getOperation();
    XMLStreamReader xmlBody = getXmlBody(request);
    try {
      Map<String, Object> ctx = getInvocationContext(request, dispatcher);
      return client.invoke(getInvocationOperation(ope), new Object[] {xmlBody}, ctx, exchange);
    } catch (SoapFault sf) {
      throw new SoapFaultException(sf);
    } catch (Fault f) {
      if (f.getMessage().contains("COULD_NOT_READ_XML")) {
        throw new BadRequestException("Error consuming the operation [" + ope + "], the request body is not a valid XML");
      }
      throw new SoapFaultException(f);
    } catch (OperationNotFoundException e) {
      String errorMsg = "The provided operation [" + ope + "] does not exist in the WSDL file [" + wsdlModel.getLocation() + "]";
      throw new BadRequestException(errorMsg, e);
    } catch (DispatcherException e) {
      throw e;
    } catch (Exception e) {
      throw new RuntimeException("Unexpected error while consuming the web service operation [" + ope + "]", e);
    }
  }

  private Object[] invoke(String operation, TransportResponse response, Exchange exchange) {
    try (ReplyHandler handler = new ReplyHandler(client.getBus(), client.getEndpoint(), client.getConduit())) {
      return handler.invoke(exchange, response);
    } catch (SoapFault sf) {
      if (sf.getMessage().contains("Error reading XMLStreamReader")) {
        badResponseException(operation, sf);
      }
      throw new SoapFaultException(sf);
    } catch (Fault f) {
      if (f.getCause() != null && f.getCause() instanceof IOException) {
        return badResponseException(operation, f);
      }
      throw new SoapFaultException(f);
    } catch (Exception e) {
      throw new RuntimeException("Unexpected error while handling the reply of the web service operation [" + operation + "]", e);
    }
  }

  private Object[] badResponseException(String operation, Fault f) {
    throw new BadResponseException("Error processing the response of the operation [" + operation
        + "], check if the content-type is correct.", f);
  }

  private XMLStreamReader getXmlBody(SoapRequest request) {
    return requestGenerator.generate(request.getOperation(), request.getContent(), getEncoding(request),
                                     request.getAttachments());
  }

  private BindingOperationInfo getInvocationOperation(String operationName) {
    // Normally its not this hard to invoke the CXF Client, but we're
    // sending along some exchange properties, so we need to use a more advanced
    // method
    Endpoint ep = client.getEndpoint();
    // The operation is always named invoke because hits our ProxyService implementation.
    String method = operationResolver.resolve(operationName).getType().equals(OperationType.ONE_WAY) ? "invokeOneWay" : "invoke";
    QName q = new QName(ep.getService().getName().getNamespaceURI(), method);
    BindingOperationInfo bop = ep.getBinding().getBindingInfo().getOperation(q);
    if (bop.isUnwrappedCapable()) {
      bop = bop.getUnwrappedOperation();
    }
    return bop;
  }

  private Map<String, Object> getInvocationContext(SoapRequest request, TransportDispatcher dispatcher) {
    Map<String, Object> props = new HashMap<>();
    OperationModel operation = operationResolver.resolve(request.getOperation());

    // is NOT mtom the attachments must not be touched by cxf, we create a custom request embedding the attachment in the xml
    props.put(MULE_ATTACHMENTS_KEY, buildCxfAttachments(request.getAttachments()));
    props.put(MULE_WSC_ADDRESS, address);
    props.put(ENCODING, getEncoding(request));
    props.put(MULE_HEADERS_KEY, buildCxfHeaders(request.getSoapHeaders()));
    props.put(MULE_TRANSPORT_HEADERS_KEY, request.getTransportHeaders());
    props.put(MESSAGE_DISPATCHER, dispatcher);
    props.put(MULE_SOAP_OPERATION_STYLE, operation.getType());
    String soapAction = operation.getSoapAction();
    if (soapAction != null) {
      props.put(MULE_SOAP_ACTION, soapAction);
    }
    props.put(FORCE_START_DOCUMENT, request.isUseXMLInitialDeclaration());

    Map<String, Object> ctx = new HashMap<>();
    ctx.put(Client.REQUEST_CONTEXT, props);
    return ctx;
  }

  private String getEncoding(SoapRequest request) {
    try {
      return new ContentType(request.getContentType()).getParameter("charset");
    } catch (ParseException e) {
      return isBlank(encoding) ? "UTF-8" : encoding;
    }
  }

  protected abstract Map<String, Attachment> buildCxfAttachments(Map<String, SoapAttachment> attachments);

  private List<SoapHeader> buildCxfHeaders(Map<String, String> headers) {
    if (headers == null) {
      return emptyList();
    }
    return headers.entrySet().stream()
        .map(header -> {
          try {
            return new SoapHeader(new QName(null, header.getKey()), stringToDomElement(header.getValue()));
          } catch (Exception e) {
            throw new BadRequestException("Cannot parse input header [" + header.getKey() + "]", e);
          }
        })
        .collect(toList());
  }

  private class ReplyHandler extends ClientImpl {

    private final Charset DEFAULT_ENCODING = UTF_8;

    public ReplyHandler(Bus b, Endpoint e, Conduit c) {
      super(b, e, c);
    }

    public Object[] invoke(Exchange exchange, TransportResponse response) throws Exception {
      initExchange(exchange, response.getHeaders(), response.getAdditionalData());

      Message message = newMessage(response.getContentType(), response.getContent());
      message.setExchange(exchange);

      getConduit().getMessageObserver().onMessage(message);

      Exception ex = this.getException(exchange);
      if (ex != null) {
        throw ex;
      } else {
        List<Object> list = exchange.getInMessage().getContent(List.class);
        return list != null ? list.toArray() : null;
      }
    }

    private Exchange initExchange(Exchange exchange, Map<String, String> headers, Map<String, String> additionalData) {

      // Init exchange
      exchange.put(Bus.class, this.getBus());
      exchange.put(Service.class, this.getEndpoint().getService());
      exchange.put(AbstractSoapCxfClient.MULE_TRANSPORT_HEADERS_KEY, headers);
      exchange.put(AbstractSoapCxfClient.MULE_TRANSPORT_ADDITIONAL_DATA_KEY, additionalData);
      exchange.put(AbstractSoapCxfClient.MULE_ATTACHMENTS_KEY, emptyMap());
      exchange.setOutMessage(new MessageImpl());

      // This needs to be set because we want the wsc closes the final stream,
      // otherwise cxf will close it too early when handling message in the StaxInEndingInterceptor.
      exchange.put(STAX_IN_NOCLOSE, TRUE);

      return exchange;
    }

    private Message newMessage(String contentType, InputStream content) {

      Object responseEncoding = resolveEncoding(contentType).orElse(DEFAULT_ENCODING.name());

      Message message = new MessageImpl();
      message.put(CONTENT_TYPE, contentType);
      message.put(ENCODING, responseEncoding);
      message.setContent(InputStream.class, content);

      return message;
    }

    private Optional<Object> resolveEncoding(String contentType) {
      try {
        final MediaType mediaType = MediaType.parse(contentType);
        return mediaType.charset().isPresent() ? Optional.of(mediaType.charset().get().name()) : Optional.empty();
      } catch (IllegalArgumentException e) {
        return Optional.empty();
      }
    }

    @Override
    public void close() throws Exception {}
  }
}
