/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator;

import org.mule.soap.api.client.BadRequestException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.internal.generator.attachment.AttachmentRequestEnricher;
import org.mule.soap.internal.util.OperationResolver;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.util.Map;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * Generates a XML SOAP request used to invoke CXF.
 * <p>
 * If no body is provided will try to generate a default one.
 * <p>
 * for each attachment will add a node with the required information depending on the protocol that it's being used.
 *
 * @since 1.0
 */
public final class SoapRequestGenerator {

  private final EmptyRequestGenerator emptyRequestGenerator;
  private final AttachmentRequestEnricher requestEnricher;

  public SoapRequestGenerator(AttachmentRequestEnricher requestEnricher, OperationResolver port) {
    this.requestEnricher = requestEnricher;
    this.emptyRequestGenerator = new EmptyRequestGenerator(port);
  }

  /**
   * Generates an {@link XMLStreamReader} SOAP request ready to be consumed by CXF.
   * @param operation   the name of the operation being invoked.
   * @param body        the body content provided by the user.
   * @param attachments the attachments provided by the user.
   */
  public XMLStreamReader generate(String operation, InputStream body, String encoding, Map<String, SoapAttachment> attachments) {
    try {
      String requestEncoding = isBlank(encoding) ? UTF_8.name() : encoding;

      body = checkEmptyStream(body);

      if (body == null) {
        body = new ByteArrayInputStream(emptyRequestGenerator.generateRequest(operation).getBytes(requestEncoding));
      }

      return requestEnricher.enrichRequest(operation, body, requestEncoding, attachments);

    } catch (XMLStreamException | IOException e) {
      throw new BadRequestException("Error generating SOAP request", e);
    }
  }

  /**
   * Checks if the input stream is empty. Returns a reference to an InputStream if it is not. Null if empty.
   */
  private InputStream checkEmptyStream(InputStream inputStream) throws IOException {
    if (inputStream == null) {
      return null;
    }

    PushbackInputStream pushbackInputStream = new PushbackInputStream(inputStream);
    int b;
    b = pushbackInputStream.read();
    if (b == -1) {
      return null;
    }
    pushbackInputStream.unread(b);
    return pushbackInputStream;
  }
}
