/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.rm;

import org.mule.soap.api.exception.ReliableMessagingStoreException;

import java.io.Serializable;
import java.util.Map;

/**
 * Store to persists reliable messaging sequences and messages.
 *
 * @param <T> the generic type of the objects to be stored
 *
 * @since 1.6
 */
public interface ReliableMessagingStore<T extends Serializable> {

  /**
   * Store the given Object.
   *
   * @param key   the identifier for <code>value</code>
   * @param value the Object to store with <code>key</code>
   * @throws ReliableMessagingStoreException if the given key cannot be stored or is <code>null</code> or if an attempt
   *                                         is made to store an object for a key that already has an object associated
   *                                         or if the store is not available or any other implementation-specific error
   *                                         occurred.
   */
  void store(String key, T value) throws Exception;

  /**
   * Updates the given Object on the given key
   *
   * @param key   the identifier for <code>value</code>
   * @param value the Object to store with <code>key</code>
   * @throws ReliableMessagingStoreException if the given Object cannot be updated or if the store is not available if
   *                                         no value for the given key was previously stored or any other
   *                                         implementation-specific error occurred.
   */
  void update(String key, T value) throws Exception;

  /**
   * * Retrieve the object stored under the given {@code key}
   *
   * @param key the identifier of the object to retrieve.
   * @return the object associated with the given key. If no object for the given key was found this returns
   *         <code>null</code>.
   * @throws ReliableMessagingStoreException if the given key is <code>null</code> or if the store is not available or
   *                                         any other implementation-specific error occurred.
   */
  T retrieve(String key) throws Exception;

  /**
   * Remove the object with the given {@code key}
   *
   * @param key the identifier of the object to remove.
   * @return the object that was previously stored for the given key
   * @throws ReliableMessagingStoreException if the given key is <code>null</code> or if no value for the given key was
   *                                         previously stored or if the store is not available or any other
   *                                         implementation-specific error occurred.
   */
  T remove(String key) throws Exception;

  /**
   * @return All the key-value pairs that this object store currently holds.
   *
   * @throws ReliableMessagingStoreException if an exception occurred while collecting the values.
   */
  Map<String, T> retrieveAll() throws Exception;

}
