/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static javax.xml.soap.SOAPConstants.URI_NS_SOAP_1_1_ENVELOPE;
import static javax.xml.soap.SOAPConstants.URI_NS_SOAP_1_2_ENVELOPE;
import static javax.xml.stream.XMLOutputFactory.IS_REPAIRING_NAMESPACES;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_ATTACHMENTS_KEY;
import static org.slf4j.LoggerFactory.getLogger;

import com.google.common.collect.ImmutableMap;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.utils.MetadataTypeUtils;
import org.mule.soap.api.exception.EncodingException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.internal.util.CopyStream;
import org.mule.wsdl.parser.model.operation.OperationModel;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import com.ctc.wstx.stax.WstxInputFactory;
import org.apache.cxf.message.Exchange;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

/**
 * Abstract implementation for a request enricher that extracts the attachments from a SOAP Response.
 *
 * @since 1.0
 */
public abstract class AttachmentResponseEnricher {

  private static final Logger LOGGER = getLogger(AttachmentResponseEnricher.class);
  private static final AtomicBoolean alreadyLoggedWarning = new AtomicBoolean();
  private static final XMLInputFactory xmlInputFactory = new WstxInputFactory();
  private static final XMLOutputFactory xmlOutputFactory = XMLOutputFactory.newInstance();
  private static final String ENVELOPE_BODY_TAG = "Body";

  static {
    xmlOutputFactory.setProperty(IS_REPAIRING_NAMESPACES, true);
  }

  private final Map<String, OperationModel> operations;

  AttachmentResponseEnricher(List<OperationModel> operations) {
    this.operations = operations.stream().collect(toMap(OperationModel::getName, identity(), (op1, op2) -> {
      if (alreadyLoggedWarning.compareAndSet(false, true)) {
        LOGGER.warn("Overloaded operations with different outputs are not fully supported");
      }
      return op1;
    }));
  }

  /**
   * Modifies the SOAP response to avoid attachment content in the response body and make decouple the attachment handling so
   * the user can have a better experience.
   */
  public InputStream enrich(XMLStreamReader xmlStreamReader, String operation, Exchange exchange)
      throws XMLStreamException {

    MetadataType attachments = operations.get(operation).getOutputType().getAttachments();

    if (attachments instanceof ObjectType) {
      Collection<ObjectFieldType> attachmentsFields = ((ObjectType) attachments).getFields();
      if (!attachmentsFields.isEmpty()) {
        return processXmlStream(xmlStreamReader, attachmentsFields, exchange);
      }
    }

    return processXmlStream(xmlStreamReader, emptySet(), exchange);
  }

  /**
   * Creates an InputStream from the response XMLStreamReader.
   * Returns an InputStream containing the response without the envelope.
   * If needed, the attachments will be added to the provided exchange.
   *
   * @param xmlStreamReader       The response from the server
   * @param attachmentsFields     The attachment fields
   * @param exchange              The exchange where to add the attachments if needed
   * @return  An InputStream containing the response with it's envelope removed.
   */
  private InputStream processXmlStream(XMLStreamReader xmlStreamReader,
                                       Collection<ObjectFieldType> attachmentsFields,
                                       Exchange exchange)
      throws XMLStreamException {
    List<String> attachmentNames = attachmentsFields.stream().map(MetadataTypeUtils::getLocalPart).collect(toList());
    XmlEventEnricher eventEnricher = attachmentNames.isEmpty() ? new VoidXmlEventEnricher() : new XmlEventEnricher();

    XMLEventReader xmlEventReader = null;
    XMLEventWriter xmlEventWriter = null;

    CopyStream outputByteStream = new CopyStream();

    try {
      xmlEventReader = xmlInputFactory.createXMLEventReader(xmlStreamReader);
      xmlEventWriter = xmlOutputFactory.createXMLEventWriter(outputByteStream);

      while (xmlEventReader.hasNext()) {
        XMLEvent xmlEvent = xmlEventReader.nextEvent();

        if (xmlEvent.isEndElement() &&
            xmlEvent.asEndElement().getName().getLocalPart().equals(ENVELOPE_BODY_TAG) &&
            (xmlEvent.asEndElement().getName().getNamespaceURI().equals(URI_NS_SOAP_1_1_ENVELOPE) ||
                xmlEvent.asEndElement().getName().getNamespaceURI().equals(URI_NS_SOAP_1_2_ENVELOPE))) {
          break;
        }

        XMLEvent enrichedEvent = eventEnricher.enrich(xmlEvent, attachmentNames, exchange);

        if (enrichedEvent != null) {
          xmlEventWriter.add(enrichedEvent);
        }
      }
      xmlEventWriter.flush();

    } finally {
      try {
        xmlStreamReader.close();
      } catch (XMLStreamException e) {
        LOGGER.warn("Could not close XMLStreamReader", e);
      }

      try {
        if (xmlEventReader != null) {
          xmlEventReader.close();
        }
      } catch (XMLStreamException e) {
        LOGGER.warn("Could not close XMLEventReader", e);
      }

      try {
        if (xmlEventWriter != null) {
          xmlEventWriter.close();
        }
      } catch (XMLStreamException e) {
        LOGGER.warn("Could not close XMLEventWriter", e);
      }
    }

    return outputByteStream.toInputStream();
  }

  /**
   * Processes the attachments nodes in the response.
   */
  protected void processAttachmentData(String attachmentName, String attachmentData, Exchange exchange) {
    ImmutableMap.Builder<String, SoapAttachment> builder = ImmutableMap.builder();

    if (exchange.get(MULE_ATTACHMENTS_KEY) != null) {
      builder.putAll((Map<String, SoapAttachment>) exchange.get(MULE_ATTACHMENTS_KEY));
    }

    builder.put(attachmentName, buildAttachment(attachmentName, attachmentData));

    exchange.put(MULE_ATTACHMENTS_KEY, builder.build());
  }

  private SoapAttachment buildAttachment(String attachmentName, String attachmentData) {
    InputStream decodedAttachment = decodeAttachment(attachmentName, attachmentData);
    return new SoapAttachment(decodedAttachment, "*/*");
  }

  /**
   * Decodes the attachment content from base64.
   */
  private InputStream decodeAttachment(String name, String attachmentContent) {
    try {
      return new ByteArrayInputStream(Base64.getDecoder().decode(attachmentContent));
    } catch (Exception e) {
      throw new EncodingException(format("Cannot decode base64 attachment [%s]", name));
    }
  }

  private class XmlEventEnricher {

    private boolean processingAttachmentElement;
    private String attachmentName;
    private StringBuilder attachmentBuffer = new StringBuilder();

    /**
     * Processes an XMLEvent. Enriches it if the event corresponds to an attachment.
     * Processing of this event content is delegated to the extending classes.
     * @param xmlEvent        Event to be processed.
     * @param attachmentNames The names of the attachments that should be processed.
     * @param exchange        The exchange where to add the extracted attachments.
     * @return Null if the event corresponds to an attachment that must be processed. Else, the provided XMLEvent.
     */
    @Nullable
    protected XMLEvent enrich(XMLEvent xmlEvent, List<String> attachmentNames, Exchange exchange) {

      if (processingAttachmentElement && xmlEvent.isEndElement()) {
        EndElement startElement = xmlEvent.asEndElement();
        String localPart = startElement.getName().getLocalPart();
        if (localPart.equals(attachmentName)) {
          if (attachmentBuffer.length() != 0) {
            processAttachmentData(attachmentName, attachmentBuffer.toString(), exchange);
            //The data is cleaned in case another attachment comes
            attachmentBuffer.setLength(0);
          }
          processingAttachmentElement = false;
          attachmentName = null;
        }
        return null;
      } else if (processingAttachmentElement) {
        if (xmlEvent.isCharacters()) {
          attachmentBuffer.append(xmlEvent.asCharacters().getData());
        }
        return null;
      } else if (xmlEvent.isStartElement()) {
        StartElement startElement = xmlEvent.asStartElement();
        String localPart = startElement.getName().getLocalPart();
        if (attachmentNames.contains(localPart)) {
          processingAttachmentElement = true;
          attachmentName = localPart;
          return null;
        }
      }

      return xmlEvent;
    }
  }

  private class VoidXmlEventEnricher extends XmlEventEnricher {

    @Override
    protected XMLEvent enrich(XMLEvent xmlEvent, List<String> attachmentNames, Exchange exchange) {
      return xmlEvent;
    }
  }
}
