/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import org.mule.soap.api.exception.EncodingException;
import org.mule.soap.api.message.SoapAttachment;

import java.io.InputStream;
import java.util.Base64;

import static org.apache.commons.io.IOUtils.toByteArray;

import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLStreamException;

/**
 * {@link AttachmentRequestEnricher} implementation to send attachments using SOAP with Attachments.
 *
 * @since 1.0
 */
public final class SoapAttachmentRequestEnricher extends AttachmentRequestEnricher {

  private static final XMLEventFactory XML_EVENT_FACTORY = XMLEventFactory.newFactory();

  /**
   * {@inheritDoc}
   * <p>
   * Creates the attachment content encoded to Base64 plain in the XML Request in the generated attachment node.
   */
  @Override
  protected void writeAttachmentContent(XMLEventWriter xmlEventWriter, String attachmentName, SoapAttachment soapAttachment)
      throws XMLStreamException {
    xmlEventWriter.add(XML_EVENT_FACTORY.createCharacters(toBase64(soapAttachment.getContent())));
  }

  /**
   * Returns an {@link String} with the content of the attachment encoded to Base64.
   *
   * @param content the content of the attachment.
   */
  private String toBase64(InputStream content) {
    try {
      return Base64.getEncoder().encodeToString(toByteArray(content));
    } catch (Exception e) {
      throw new EncodingException("Could not encode attachment content to base64", e);
    }
  }
}
