/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.interceptor.gzip;

import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.interceptor.MessageSenderInterceptor;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.Phase;
import org.mule.soap.internal.util.DelegatingOutputStream;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.zip.GZIPOutputStream;

/**
 * This interceptor is responsible from compressing a message when the Content-Encoding is set to gzip or x-gzip. It won't set the
 * outbound property since it requires that it's previously set. This makes sense since we are proxying.
 */
public class ProxyGZIPOutInterceptor extends AbstractProxyGZIPInterceptor {

  public ProxyGZIPOutInterceptor() {
    super(Phase.PREPARE_SEND);
    addAfter(MessageSenderInterceptor.class.getName());
  }

  @Override
  public void handleMessage(Message message) throws Fault {
    if (isEncoded(message)) {
      DelegatingOutputStream os = message.getContent(DelegatingOutputStream.class);

      if (os == null) {
        return;
      }

      try {
        ByteArrayOutputStream baos = new ByteArrayOutputStream(os.getOutputStream().size());
        GZIPOutputStream gzipOutputStream = new GZIPOutputStream(baos);

        gzipOutputStream.write(os.getOutputStream().toByteArray());
        gzipOutputStream.close();

        message.getContent(DelegatingOutputStream.class).setOutputStream(baos);
      } catch (IOException io) {
        throw new Fault(io);
      }
    }
  }

}
