/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.client;

import org.mule.soap.api.message.SoapRequest;
import org.mule.soap.api.message.SoapResponse;
import org.mule.soap.api.rm.CreateSequenceRequest;
import org.mule.soap.api.rm.TerminateSequenceRequest;
import org.mule.soap.api.transport.TransportDispatcher;
import org.mule.soap.api.transport.TransportResponse;
import org.mule.soap.internal.message.EmptySoapResponse;

import static java.util.Collections.emptyMap;

/**
 * Contract for clients that consumes SOAP Web Services.
 *
 * @since 1.0
 */
public interface SoapClient {

  /**
   * Sends a {@link SoapRequest} blocking the current thread until a response is available or the request times out.
   *
   * @param request a {@link SoapRequest} instance.
   * @param dispatcher a {@link TransportDispatcher} that will be used to dispatch the {@link SoapRequest}
   * @return a {@link SoapResponse} instance with the XML content and Headers if any.
   */
  SoapResponse consume(SoapRequest request, TransportDispatcher dispatcher) throws BadRequestException, BadResponseException;


  /**
   * Generates a {@link SoapResponse} from a {@link TransportResponse}
   *
   * @param operation The name of the operation related to the response
   * @param response a {@link TransportResponse} instance
   * @return a {@link SoapResponse} instance with the XML content and Headers if any.
   *
   * @since 1.5
   */
  default SoapResponse parseResponse(String operation, TransportResponse response) throws BadResponseException {
    return new EmptySoapResponse(emptyMap(), emptyMap());
  }

  /**
   * Shuts down the client and all resources that is holding.
   */
  void destroy();

  /**
   * Creates a Reliable Message sequence
   *
   * @param request a {@link CreateSequenceRequest} instance.
   * @param dispatcher a {@link TransportDispatcher} that will be used to dispatch the RM protocol message
   * @return a {@code String} representing a reliable message sequence identifier
   *
   * @since 1.6
   */
  default String createSequence(CreateSequenceRequest request, TransportDispatcher dispatcher) {
    return null;
  }

  /**
   * Terminates a Reliable Message sequence
   *
   * @param request a {@link TerminateSequenceRequest} instance.
   * @param dispatcher a {@link TransportDispatcher} that will be used to dispatch the RM protocol message
   *
   * @since 1.6
   */
  default void terminateSequence(TerminateSequenceRequest request, TransportDispatcher dispatcher) {}
}
