/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.interceptor;

import static org.mule.soap.internal.rm.RMUtils.copySettings;

import static java.lang.Boolean.TRUE;
import static java.util.Collections.emptyList;
import static org.apache.cxf.interceptor.StaxInEndingInterceptor.STAX_IN_NOCLOSE;
import static org.apache.cxf.message.Message.CONTENT_TYPE;
import static org.apache.cxf.message.Message.ENCODING;
import static org.apache.cxf.phase.Phase.SEND_ENDING;

import org.mule.soap.api.SoapWebServiceConfiguration;
import org.mule.soap.api.transport.TransportDispatcher;
import org.mule.soap.api.transport.TransportResponse;
import org.mule.soap.internal.client.AbstractSoapCxfClient;
import org.mule.wsdl.parser.model.operation.OperationType;

import java.io.InputStream;
import java.util.Optional;

import com.google.common.net.MediaType;
import org.apache.cxf.endpoint.ClientImpl;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.apache.cxf.message.MessageImpl;
import org.apache.cxf.phase.AbstractPhaseInterceptor;
import org.apache.cxf.transport.MessageObserver;

/**
 * CXF interceptor that uses a custom {@link TransportDispatcher}, specified in the {@link SoapWebServiceConfiguration} to send a
 * SOAP message and inject the obtained response into the CXF <strong>in</strong> (response) interceptors lifecycle.
 *
 * @since 1.0
 */
public class MessageDispatcherInterceptor extends AbstractPhaseInterceptor<Message> {

  private final MessageObserver messageObserver;

  public MessageDispatcherInterceptor(MessageObserver messageObserver) {
    super(SEND_ENDING);
    this.messageObserver = messageObserver;
  }

  /**
   * {@inheritDoc}
   * <p>
   * Intercepts the SOAP message and performs the dispatch of it, receiving the response and
   * sending it to the IN intercepting processor chain.
   */
  @Override
  public void handleMessage(Message message) throws Fault {
    Exchange exchange = message.getExchange();
    Object encoding = exchange.get(ENCODING);
    message.put(ENCODING, encoding);

    // Performs all the remaining interceptions before sending.
    message.getInterceptorChain().doIntercept(message);

    // Wipe the request attachment list, so don't get mixed with the response ones.
    message.setAttachments(emptyList());

    TransportDispatcher dispatcher = (TransportDispatcher) exchange.get(AbstractSoapCxfClient.MESSAGE_DISPATCHER);
    TransportResponse response = dispatcher.dispatch(TransportRequestFactory.createDispatchingRequest(message));

    // This needs to be set because we want the wsc closes the final stream,
    // otherwise cxf will close it too early when handling message in the StaxInEndingInterceptor.
    exchange.put(STAX_IN_NOCLOSE, TRUE);

    if (OperationType.ONE_WAY.equals(exchange.get(AbstractSoapCxfClient.MULE_SOAP_OPERATION_STYLE))) {
      exchange.put(ClientImpl.FINISHED, true);
    } else {
      handleRequestResponse(exchange, encoding, response, message);
    }
  }

  private void handleRequestResponse(Exchange exchange, Object requestEncoding, TransportResponse response, Message outMessage) {
    Message inMessage = new MessageImpl();

    String contentType = response.getContentType();
    // Extract the encoding from the response content-type if present, otherwise use the encoding from the request
    Object responseEncoding = resolveEncoding(contentType).orElse(requestEncoding);

    inMessage.put(CONTENT_TYPE, contentType);
    inMessage.put(ENCODING, responseEncoding);
    inMessage.setContent(InputStream.class, response.getContent());
    exchange.put(AbstractSoapCxfClient.MULE_TRANSPORT_HEADERS_KEY, response.getHeaders());
    exchange.put(AbstractSoapCxfClient.MULE_TRANSPORT_ADDITIONAL_DATA_KEY, response.getAdditionalData());
    inMessage.setExchange(exchange);
    copySettings(outMessage, inMessage);
    messageObserver.onMessage(inMessage);
  }

  private Optional<Object> resolveEncoding(String contentType) {
    try {
      final MediaType mediaType = MediaType.parse(contentType);
      return mediaType.charset().isPresent() ? Optional.of(mediaType.charset().get().name()) : Optional.empty();
    } catch (IllegalArgumentException e) {
      return Optional.empty();
    }
  }
}
