package org.mule.soap.api.security;

import com.google.common.collect.ImmutableMap;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import static java.util.Optional.empty;
import static org.apache.cxf.common.util.StringUtils.isEmpty;
import static org.apache.wss4j.common.ConfigurationConstants.*;

/**
 * Adds global configurations to the outgoing SOAP message.
 *
 * @since 1.4
 */
public class WssGlobalOutgoingSecurityStrategy implements SecurityStrategy {

  /**
   * A SOAP message may travel from a sender to a receiver by passing different endpoints along the message path. The SOAP actor
   * attribute is used to address the {@code wsse:Security} element to a specific endpoint.
   */
  private String actor;

  /**
   * Whether mustUnderstand attribute in {@code wsse:Security} header is true or false.
   */
  private boolean mustUnderstand;


  public WssGlobalOutgoingSecurityStrategy(String actor, boolean mustUnderstand) {
    this.actor = actor;
    this.mustUnderstand = mustUnderstand;
  }

  @Override
  public SecurityStrategyType securityType() {
    return SecurityStrategyType.OUTGOING;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return empty();
  }

  @Override
  public String securityAction() {
    return "";
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    ImmutableMap.Builder<String, Object> builder = ImmutableMap.<String, Object>builder()
        .put(MUST_UNDERSTAND, String.valueOf(mustUnderstand));

    if (!isEmpty(actor)) {
      builder.put(ACTOR, actor);
    }

    builder.put(STORE_BYTES_IN_ATTACHMENT, "false");

    return builder.build();
  }
}
