/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import org.mule.soap.api.client.BadRequestException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.internal.util.CopyStream;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashSet;
import java.util.Map;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.events.XMLEvent;

import org.slf4j.Logger;

/**
 * Abstract implementation for a request enricher that adds a node for each sent attachment to the incoming SOAP request with all
 * the information required to send the attachments using the SOAP protocol.
 *
 * @since 1.0
 */
public abstract class AttachmentRequestEnricher {

  private static final Logger LOGGER = getLogger(AttachmentRequestEnricher.class);
  private static final XMLInputFactory XML_INPUT_FACTORY = XMLInputFactory.newInstance();
  private static final XMLOutputFactory XML_OUTPUT_FACTORY = XMLOutputFactory.newInstance();
  private static final XMLEventFactory XML_EVENT_FACTORY = XMLEventFactory.newInstance();

  /**
   * @param body        the XML SOAP body provided by the user.
   * @param attachments the attachments to upload.
   */
  public XMLStreamReader enrichRequest(String operation, InputStream body, String encoding,
                                       Map<String, SoapAttachment> attachments)
      throws XMLStreamException {

    CopyStream outputByteStream = new CopyStream();

    XMLEventReader xmlEventReader = null;
    XMLEventWriter xmlEventWriter = null;

    try {
      try {
        xmlEventReader = XML_INPUT_FACTORY.createXMLEventReader(body, encoding);
        xmlEventWriter = XML_OUTPUT_FACTORY.createXMLEventWriter(outputByteStream);

        LinkedHashSet<String> addedAttachments = new LinkedHashSet<>();

        while (xmlEventReader.hasNext()) {
          XMLEvent xmlEvent = xmlEventReader.nextEvent();

          if (xmlEvent.isEndElement() && xmlEvent.asEndElement().getName().getLocalPart().equals(operation)) {
            for (String key : attachments.keySet()) {
              if (!addedAttachments.contains(key)) {
                writeAttachmentStart(xmlEventWriter, key);
                writeAttachmentContent(xmlEventWriter, key, attachments.get(key));
                writeAttachmentEnd(xmlEventWriter, key);
              }
            }
            xmlEventWriter.add(xmlEvent);
          } else {
            String attachmentName = getAttachmentName(xmlEvent);

            if (isNotBlank(attachmentName) && !addedAttachments.contains(attachmentName)
                && attachments.containsKey(attachmentName)) {
              writeAttachmentContent(xmlEventWriter, attachmentName, attachments.get(attachmentName));
              addedAttachments.add(attachmentName);
            } else {
              xmlEventWriter.add(xmlEvent);
            }
          }
        }
        xmlEventWriter.flush();

      } finally {
        try {
          body.close();
        } catch (IOException e) {
          LOGGER.warn("Could not close body InputStream", e);
        }

        try {
          if (xmlEventWriter != null) {
            xmlEventWriter.close();
          }
        } catch (XMLStreamException e) {
          LOGGER.warn("Could not close XMLEventWriter", e);
        }

        try {
          if (xmlEventReader != null) {
            xmlEventReader.close();
          }
        } catch (XMLStreamException e) {
          LOGGER.warn("Could not close XMLEventReader", e);
        }
      }
    } catch (XMLStreamException e) {
      throw new BadRequestException(format("Error consuming the operation [%s], the request body is not a valid XML", operation),
                                    e);
    }

    return XML_INPUT_FACTORY.createXMLStreamReader(outputByteStream.toInputStream());
  }

  private void writeAttachmentStart(XMLEventWriter xmlEventWriter, String key) throws XMLStreamException {
    xmlEventWriter.add(XML_EVENT_FACTORY.createStartElement(EMPTY, EMPTY, key));
  }

  private void writeAttachmentEnd(XMLEventWriter xmlEventWriter, String key) throws XMLStreamException {
    xmlEventWriter.add(XML_EVENT_FACTORY.createEndElement(EMPTY, EMPTY, key));
  }

  protected abstract void writeAttachmentContent(XMLEventWriter xmlEventWriter, String attachmentName,
                                                 SoapAttachment soapAttachment)
      throws XMLStreamException;

  private String getAttachmentName(XMLEvent event) {
    if (event.isCharacters()) {
      String data = event.asCharacters().getData();
      if (data.startsWith("cid:")) {
        return data.substring(4);
      }
    }
    return null;
  }

}
