/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.interceptor;

import static java.lang.String.CASE_INSENSITIVE_ORDER;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.cxf.message.Message.CONTENT_TYPE;
import static org.apache.cxf.message.Message.ENCODING;
import static org.mule.soap.internal.interceptor.SoapActionInterceptor.SOAP_ACTION;

import org.mule.soap.api.transport.TransportRequest;
import org.mule.soap.internal.client.AbstractSoapCxfClient;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.Map;
import java.util.TreeMap;

import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;

import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.mule.soap.internal.util.DelegatingOutputStream;

/**
 * Creates {@link TransportRequest} instances given a {@link Message}.
 *
 * @since 1.1
 */
public class TransportRequestFactory {

  public static TransportRequest createDispatchingRequest(Message message) {
    Exchange exchange = message.getExchange();

    String action = (String) exchange.get(AbstractSoapCxfClient.MULE_SOAP_ACTION);
    String contentType = getContentType(message);

    Map<String, String> headers = new TreeMap<>(CASE_INSENSITIVE_ORDER);
    headers.put(SOAP_ACTION, action);
    headers.put(CONTENT_TYPE, contentType);
    headers.putAll((Map) exchange.get(AbstractSoapCxfClient.MULE_TRANSPORT_HEADERS_KEY));

    ByteArrayOutputStream outputStream = message.getContent(DelegatingOutputStream.class).getOutputStream();
    InputStream content = new ByteArrayInputStream(outputStream.toByteArray());

    return new TransportRequest(content, (String) exchange.get(AbstractSoapCxfClient.MULE_WSC_ADDRESS), headers);
  }

  private static String getContentType(Message message) {
    String contentTypeString = (String) message.get(CONTENT_TYPE);

    if (!isBlank(contentTypeString)) {
      try {
        ContentType contentType = new ContentType(contentTypeString);
        String charset = (String) message.get(ENCODING);
        if (isXMLSubType(contentType.getSubType()) && !isBlank(charset)) {
          contentType.setParameter("charset", charset);
        }
        return contentType.toString();
      } catch (ParseException e) {
        return contentTypeString;
      }
    }

    return null;
  }

  private static Boolean isXMLSubType(String subType) {
    return subType.equalsIgnoreCase("xml") ||
        subType.equalsIgnoreCase("soap+xml");
  }

}
