/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.conduit;

import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_ATTACHMENTS_KEY;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_HEADERS_KEY;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MESSAGE_DISPATCHER;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_SOAP_ACTION;

import org.apache.cxf.binding.soap.SoapHeader;
import org.apache.cxf.binding.soap.SoapMessage;
import org.apache.cxf.message.Attachment;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.apache.cxf.transport.AbstractConduit;
import org.apache.cxf.ws.addressing.AddressingProperties;
import org.apache.cxf.ws.addressing.ContextUtils;
import org.apache.cxf.ws.addressing.EndpointReferenceType;
import org.apache.cxf.ws.rm.RMContextUtils;
import org.mule.soap.internal.interceptor.MessageDispatcherInterceptor;
import org.mule.soap.internal.util.DelegatingOutputStream;

import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

/**
 * A pipe between peers that channels transport-level messages.
 * <p>
 * Adds the a provided message dispatcher to the CXF execution chain.
 *
 * @since 1.0
 */
final class SoapServiceConduit extends AbstractConduit {

  private static final Logger LOGGER = Logger.getLogger(SoapServiceConduit.class.getSimpleName());

  SoapServiceConduit(EndpointReferenceType t) {
    super(t);
  }

  @Override
  protected Logger getLogger() {
    return LOGGER;
  }

  @Override
  public void prepare(Message message) {
    // Sets a new OutputStream where the message is going to be handled,
    // through a DelegatingOutputStream which allows us to recover the original one later.
    DelegatingOutputStream delegatingOutputStream = new DelegatingOutputStream();
    message.setContent(OutputStream.class, delegatingOutputStream);
    message.setContent(DelegatingOutputStream.class, delegatingOutputStream);

    enrichRMProtocolMessage(message);
    addHeaders(message);
    addAttachments(message);
    addMessageDispatcher(message);
  }

  private void enrichRMProtocolMessage(Message message) {
    AddressingProperties maps = ContextUtils.retrieveMAPs(message, false, true, false);
    if (maps == null) {
      return;
    }

    String action = null;
    if (null != maps.getAction()) {
      action = maps.getAction().getValue();
    }

    if (RMContextUtils.isRMProtocolMessage(action)) {
      Exchange exchange = message.getExchange();
      exchange.put(MULE_SOAP_ACTION, action);
    }
  }

  private void addMessageDispatcher(Message message) {
    message.getInterceptorChain().add(new MessageDispatcherInterceptor(getMessageObserver()));
  }

  private void addAttachments(Message message) {
    Map<String, Attachment> soapAttachments = (Map<String, Attachment>) message.getExchange().get(MULE_ATTACHMENTS_KEY);
    message.setAttachments(soapAttachments.values());
  }

  private void addHeaders(Message message) {
    List<SoapHeader> soapHeaders = (List<SoapHeader>) message.getExchange().get(MULE_HEADERS_KEY);
    soapHeaders.forEach(header -> ((SoapMessage) message).getHeaders().add(header));
  }

  @Override
  public void close(Message msg) throws IOException {
    OutputStream os = msg.getContent(OutputStream.class);
    if (os != null) {
      os.close();
    }
  }
}
