/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.exception;

import org.apache.cxf.binding.soap.SoapFault;
import org.apache.cxf.interceptor.Fault;
import org.mule.soap.internal.util.XmlTransformationException;
import org.mule.soap.internal.util.XmlTransformationUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import javax.xml.namespace.QName;
import java.util.Map;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * A {@link RuntimeException} that represents a SOAP Fault that occurs during invocation processing.
 *
 * @since 1.0
 */
public final class SoapFaultException extends RuntimeException {

  private static final Logger LOGGER = LoggerFactory.getLogger(SoapFaultException.class);

  private final QName faultCode;
  private final QName subCode;
  private final String detail;
  private final String reason;
  private final String node;
  private final String role;
  private Map<String, String> additionalTransportData;

  public SoapFaultException(QName code, QName subCode, String detail, String reason, String node, String role, Throwable cause) {
    super(reason, cause);
    this.faultCode = code;
    this.subCode = subCode;
    this.reason = reason;
    this.node = node;
    this.role = role;
    this.detail = detail;
  }

  public SoapFaultException(SoapFault f) {
    this(f.getFaultCode(), f.getSubCode(), parseExceptionDetail(f.getDetail()), f.getReason(), f.getNode(), f.getRole(), f);
  }

  public SoapFaultException(SoapFault f, Map<String, String> additionalTransportData) {
    this(f.getFaultCode(), f.getSubCode(), parseExceptionDetail(f.getDetail()), f.getReason(), f.getNode(), f.getRole(), f);
    this.additionalTransportData = additionalTransportData;
  }

  public SoapFaultException(Fault f) {
    this(f.getFaultCode(), null, parseExceptionDetail(f.getDetail()), "", null, null, f);
  }

  /**
   * @return a code used to indicate a class of errors.
   */
  public QName getFaultCode() {
    return faultCode;
  }

  /**
   * @return the cause (reason) of the fault.
   */
  public String getReason() {
    return reason;
  }

  /**
   * @return an Optional Subcode element if you need to break down the {@link this#getFaultCode()} into subcodes.
   */
  public Optional<QName> getSubCode() {
    return ofNullable(subCode);
  }

  /**
   * @return an element that carries application-specific error messages. It can contain child elements called detail entries.
   */
  public String getDetail() {
    return detail;
  }

  /**
   * @return an URI identifying the node in which the Fault occurred.
   */
  public Optional<String> getNode() {
    return ofNullable(node);
  }

  /**
   * @return the role of the node in which the fault occurred.
   */
  public Optional<String> getRole() {
    return ofNullable(role);
  }

  private static String parseExceptionDetail(Element detail) {
    try {
      return XmlTransformationUtils.nodeToString(detail);
    } catch (XmlTransformationException e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Error while parsing Soap Exception detail: " + detail.toString(), e);
      }
      return "";
    }
  }

  @Override
  public String toString() {
    return "SoapFault{" +
        "faultCode=" + faultCode +
        ", subCode=" + subCode +
        ", detail='" + detail + '\'' +
        ", reason='" + reason + '\'' +
        ", node='" + node + '\'' +
        ", role='" + role + '\'' +
        '}';
  }
}
