/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.interceptor.gzip;

import com.google.common.net.HttpHeaders;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Message;
import org.apache.cxf.transport.common.gzip.GZIPInInterceptor;
import org.apache.cxf.transport.common.gzip.GZIPOutInterceptor;
import org.mule.soap.internal.client.AbstractSoapCxfClient;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.zip.GZIPInputStream;

/**
 * This interceptor is responsible for decompressing a message received with Content-Encoding that includes gzip or x-zip. It
 * won't delete the Content-Encoding property since we are proxying and we might still require it.
 */
public class ProxyGZIPInInterceptor extends GZIPInInterceptor {

  @Override
  public void handleMessage(Message message) throws Fault {
    Map<String, String> headers =
        (Map<String, String>) message.getExchange().get(AbstractSoapCxfClient.MULE_TRANSPORT_HEADERS_KEY);

    if (headers != null && Optional.ofNullable(headers.get(HttpHeaders.CONTENT_ENCODING))
        .map(h -> Arrays.asList(GZIPOutInterceptor.ENCODINGS.split(h.trim())))
        .filter(encs -> encs.contains("gzip") || encs.contains("x-gzip"))
        .isPresent()) {
      InputStream is = message.getContent(InputStream.class);

      if (is == null) {
        return;
      }

      try {
        message.setContent(InputStream.class, new GZIPInputStream(is));
      } catch (IOException io) {
        throw new Fault(io);
      }
    }
  }

}
