/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.substringBetween;

import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.soap.api.client.BadRequestException;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.operation.OperationModel;

import javax.xml.namespace.QName;

/**
 * Enables the construction of request bodies for web service operations that don't require input parameters.
 *
 * @since 1.0
 */
final class EmptyRequestGenerator {

  /**
   * SOAP request mask for operations without input parameters
   */
  private static final String NO_PARAMS_SOAP_BODY_CALL_MASK = "<ns:%s xmlns:ns=\"%s\"/>";

  private PortModel portModel;

  public EmptyRequestGenerator(PortModel portModel) {
    this.portModel = portModel;
  }

  /**
   * Generates a request body for an operation that don't require input parameters, if the required XML in the body is just one
   * constant element.
   */
  String generateRequest(String operationName) {

    OperationModel operation = portModel.getOperation(operationName);
    MetadataType body = operation.getInputType().getBody();

    if (isOperationWithRequiredParameters(body)) {
      throw new BadRequestException(format("Cannot build default body request for operation [%s], "
          + "it requires input parameters", operationName));
    }

    QName typeQname = QName.valueOf(operation.getInputType().getBody()
        .getAnnotation(TypeIdAnnotation.class).map(TypeIdAnnotation::getValue)
        .orElseThrow(() -> new BadRequestException("Cannot create empty request for operation [" + operationName + "]")));

    String localPart = typeQname.getLocalPart();
    if (localPart.contains("#root:{")) {
      localPart = substringBetween(typeQname.getLocalPart(), "#root:{", "}");
      return format(NO_PARAMS_SOAP_BODY_CALL_MASK, operationName, localPart);
    }

    return format(NO_PARAMS_SOAP_BODY_CALL_MASK, typeQname.getLocalPart(), typeQname.getNamespaceURI());
  }

  private boolean isOperationWithRequiredParameters(MetadataType wrapperBodyType) {
    if (wrapperBodyType instanceof ObjectType) {
      MetadataType bodyType = ((ObjectType) wrapperBodyType).getFields().iterator().next().getValue();
      return bodyType instanceof ObjectType && !((ObjectType) bodyType).getFields().isEmpty();
    }
    return false;
  }
}
