/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.rm.store;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.apache.cxf.ws.rm.DestinationSequence;
import org.apache.cxf.ws.rm.ProtocolVariation;
import static org.mule.soap.internal.rm.RMUtils.toByteArray;

public class DestinationSequenceTransfer implements Serializable {

  private String addressValue;
  private long lastMessageNumber;
  private String endpointIdentifier;
  private byte[] acknowledged;
  private boolean terminate;
  private String protocolVersion;
  private List<MessageTransfer> messageTransfers;

  public DestinationSequenceTransfer(DestinationSequence destinationSequence) throws IOException {
    this.addressValue = destinationSequence.getAcksTo().getAddress().getValue();
    this.lastMessageNumber = destinationSequence.getLastMessageNumber();
    this.endpointIdentifier = getEndpointIdentifier(destinationSequence);
    this.acknowledged = toByteArray(destinationSequence.getAcknowledgment());
    this.terminate = destinationSequence.isTerminated();
    this.protocolVersion = encodeProtocolVersion(destinationSequence.getProtocol());
    this.messageTransfers = new ArrayList<>();
  }

  public String getAddressValue() {
    return addressValue;
  }

  public void setAddressValue(String addressValue) {
    this.addressValue = addressValue;
  }

  public long getLastMessageNumber() {
    return lastMessageNumber;
  }

  public void setLastMessageNumber(long lastMessageNumber) {
    this.lastMessageNumber = lastMessageNumber;
  }

  public String getEndpointIdentifier() {
    return endpointIdentifier;
  }

  public void setEndpointIdentifier(String endpointIdentifier) {
    this.endpointIdentifier = endpointIdentifier;
  }

  public byte[] getAcknowledged() {
    return acknowledged;
  }

  public void setAcknowledged(byte[] acknowledged) {
    this.acknowledged = acknowledged;
  }

  public boolean isTerminate() {
    return terminate;
  }

  public void setTerminate(boolean terminate) {
    this.terminate = terminate;
  }

  public String getProtocolVersion() {
    return protocolVersion;
  }

  public void setProtocolVersion(String protocolVersion) {
    this.protocolVersion = protocolVersion;
  }

  public List<MessageTransfer> getMessageTransfers() {
    return messageTransfers;
  }

  public void setMessageTransfers(List<MessageTransfer> messageTransfers) {
    this.messageTransfers = messageTransfers;
  }

  public void addMessageTransfer(MessageTransfer messageTransfer) {
    this.messageTransfers.add(messageTransfer);
  }

  private static String encodeProtocolVersion(ProtocolVariation pro) {
    return pro.getCodec().getWSRMNamespace() + ' ' + pro.getCodec().getWSANamespace();
  }

  private String getEndpointIdentifier(DestinationSequence destinationSequence) {
    try {
      return destinationSequence.getEndpointIdentifier();
    } catch (Exception e) {
      return null;
    }
  }
}
