/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import org.mule.soap.api.message.SoapAttachment;

import static java.lang.String.format;
import static java.util.Collections.singleton;

import java.util.Iterator;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.Namespace;

/**
 * {@link AttachmentRequestEnricher} implementation for clients that works with MTOM.
 *
 * @since 1.0
 */
public final class MtomRequestEnricher extends AttachmentRequestEnricher {

  private static XMLEventFactory xmlEventFactory = XMLEventFactory.newFactory();

  private static final String XOP_NS = "http://www.w3.org/2004/08/xop/include";
  private static final String XOP_PREFIX = "xop";
  private static final String INCLUDE_LOCAL_PART = "Include";
  private static final String HREF = "href";
  private static final String CONTENT_ID_MASK = "cid:%s";

  /**
   * {@inheritDoc}
   * <p>
   * Creates the XOP element to the attachment node this way the attachments gets tracked with its CID (content id) when it's sent
   * in the multipart request.
   * <p>
   * Basically creates this content
   * {@code <xop:Include xmlns:xop="http://www.w3.org/2004/08/xop/include" href="cid:attachmentContentId"/>} and uses the <strong>
   * attachmentContentId</strong> to identify the attachment in the multipart.
   */
  @Override
  protected void writeAttachmentContent(XMLEventWriter xmlEventWriter, String attachmentName, SoapAttachment soapAttachment)
      throws XMLStreamException {

    Attribute hrefAttribute = xmlEventFactory.createAttribute(HREF, format(CONTENT_ID_MASK, attachmentName));
    Namespace xopNamespace = xmlEventFactory.createNamespace(XOP_PREFIX, XOP_NS);
    Iterator<Namespace> namespaceIterator = singleton(xopNamespace).iterator();


    xmlEventWriter.add(xmlEventFactory.createStartElement(new QName(XOP_NS, INCLUDE_LOCAL_PART, XOP_PREFIX),
                                                          singleton(hrefAttribute).iterator(),
                                                          namespaceIterator));

    xmlEventWriter.add(xmlEventFactory.createEndElement(new QName(XOP_NS, INCLUDE_LOCAL_PART, XOP_PREFIX),
                                                        namespaceIterator));
  }
}
