/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security;

import com.google.common.collect.ImmutableMap;
import org.apache.wss4j.common.crypto.Merlin;
import org.mule.soap.api.security.stores.WssStoreConfiguration;
import org.mule.soap.api.security.stores.WssTrustStoreConfiguration;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import static java.util.Optional.empty;
import static org.apache.wss4j.common.ConfigurationConstants.SIG_PROP_REF_ID;
import static org.apache.wss4j.common.crypto.Merlin.LOAD_CA_CERTS;

/**
 * Verifies the signature of a SOAP response, using certificates of the trust-store in the provided TLS context.
 *
 * @since 1.0
 */
public class WssVerifySignatureSecurityStrategy implements SecurityStrategy {

  private static final String WS_VERIFY_SIGNATURE_PROPERTIES_KEY = "verifySignatureProperties";

  /**
   * The truststore to use to verify the signature.
   */
  private final WssTrustStoreConfiguration trustStoreConfiguration;

  public WssVerifySignatureSecurityStrategy(WssTrustStoreConfiguration trustStoreConfiguration) {
    this.trustStoreConfiguration = trustStoreConfiguration;
  }

  public WssVerifySignatureSecurityStrategy() {
    this.trustStoreConfiguration = null;
  }

  @Override
  public SecurityStrategyType securityType() {
    return SecurityStrategyType.INCOMING;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return empty();
  }

  @Override
  public String securityAction() {
    return SecurityActions.SIGNATURE;
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    Properties signatureProps = trustStoreConfiguration != null ? trustStoreConfiguration.getConfigurationProperties()
        : getDefaultTrustStoreConfigurationProperties();

    return ImmutableMap.<String, Object>builder()
        .put(SIG_PROP_REF_ID, WS_VERIFY_SIGNATURE_PROPERTIES_KEY)
        .put(WS_VERIFY_SIGNATURE_PROPERTIES_KEY, signatureProps)
        .build();
  }

  private Properties getDefaultTrustStoreConfigurationProperties() {
    Properties properties = new Properties();
    properties.setProperty(WssStoreConfiguration.WS_CRYPTO_PROVIDER_KEY, Merlin.class.getCanonicalName());
    properties.setProperty(LOAD_CA_CERTS, String.valueOf(true));
    return properties;
  }
}
