/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.interceptor.gzip;

import com.google.common.net.HttpHeaders;
import org.apache.cxf.message.Message;
import org.apache.cxf.transport.common.gzip.GZIPOutInterceptor;
import org.mule.soap.internal.client.AbstractSoapCxfClient;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * This interceptor is responsible from compressing a message when the Content-Encoding is set to gzip or x-gzip. It won't set the
 * outbound property since it requires that it's previously set. This makes sense since we are proxying.
 */
public class ProxyGZIPOutInterceptor extends GZIPOutInterceptor {

  @Override
  public UseGzip gzipPermitted(Message message) {

    boolean useGzip = false;
    Map<String, String> headers =
        (Map<String, String>) message.getExchange().get(AbstractSoapCxfClient.MULE_TRANSPORT_HEADERS_KEY);

    if (headers == null) {
      return GZIPOutInterceptor.UseGzip.NO;
    }

    Optional<String> header = Optional.ofNullable(headers.get(HttpHeaders.CONTENT_ENCODING));

    if (header.isPresent()) {
      List<String> encodings = Arrays.asList(GZIPOutInterceptor.ENCODINGS.split(header.get().trim()));
      useGzip = encodings.contains("gzip") || encodings.contains("x-gzip");
    }
    return useGzip ? GZIPOutInterceptor.UseGzip.YES : GZIPOutInterceptor.UseGzip.NO;
  }


}
