/**
 * Returns a list of key value pair object describing the object entries
 *
 * .Example
 * [source]
 * ----
 * %dw 2.0
 * import dw::core::Objects
 * ---
 * Objects::entrySet({a: true, b: 1})
 * ----
 *
 * .Output
 * ----
 * [{key: "a", value: true}, {key: "b", value: 1}]
 * ----
 *
 */
fun entrySet<T <: Object>(obj: T) =
  obj pluck (value, key) -> {
    key: key,
    value: value
  }

/**
* Returns selecting only the entry at the specified index.
* If the index is not valid it return null. If index es less than 0 then it will start from the last element e.g {a: true, b: 1} entryAt -1 returns the last entry
*
* .Example
* [source]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::entryAt({a: true, b: 1}, 0)
* ----
*
* .Output
* ----
* {a: true}
* ----
*/
fun entryAt<K, V>(from:{(K): V}, index: Number): {(K)?: V} | Null =
    using(realIndex = if(index < 0) sizeOf(from) + index else index, result = from filterObject $$$ == realIndex)
        if(isEmpty(result))
            null
        else
           result

/**
* Returns the list of key names of an object
*
* .Example
* [source]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* ----
* ["a","b"]
* ----
*/
fun nameSet(obj: Object): Array<String> = obj pluck ($$ as String)

/**
* Returns the list of key names of an object
*
* .Example
* [source]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* ----
* ["a","b"]
* ----
*/
fun keySet<T <: Object>(obj: T): ? = obj pluck $$

/**
* Returns the list of key values of an object
*
* .Example
* [source]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* ----
* [true,1]
* ----
*/
fun valueSet <K,V>(obj: {K?: V}): Array<V> = obj pluck $

/**
 * Overrides the source with the target object so that the result with contain all the properties from the target
 * plus the properties on the source that are not declared on the target
 *
 * .Example
 * [source]
 * ----
 * %dw 2.0
 * import mergeWith from dw::core::Objects
 * ---
 * {a: true, b: 1} mergeWith {a: false, c: "Test"}
 * ----
 *
 * .Output
 * ----
 * {a: false, b: 1 , c: "Test"}
 * ----
 *
 */
fun mergeWith<T <: Object,V <: Object>(source: T, target: V): ? =
  (source -- keySet(target)) ++ target

/**
* Helper method to make `mergetWith` null friendly
*/
fun mergeWith<T <: Object>(a: Null, b: T): T = b

/**
* Helper method to make `mergetWith` null friendly
*/
fun mergeWith<T <: Object>(a: T, b: Null): T = a