/**
 * The functions described here are packaged in the Crypto module. The module is included with Mule runtime, but you must import it to your DataWeave code by adding the line `import dw::Crypto` to your header.
 *
 *
 * Example
 * [source]
 * ----
 * %dw 2.0
 * import dw::Crypto
 * ---
 * Crypto::MD5("asd" as Binary)
 * ----
 *
 * This module contains encrypting functions that follow common algorithms such as MD5 or SHA1.
 *
*/
%dw 2.0
/**
* Computes a Hash-based Message Authentication Code (HMAC) using the SHA1 hash function.
*/
fun HMACBinary(content: Binary, secret: Binary): Binary = native("system::HMACFunctionValue")

/**
* Computes the hash of the specified content with the given algorithm and returns the binary content.
*
* Algorithm can be "SHA-1", "MD5", "SHA-256", etc
*
*/
fun hashWith(content: Binary, algorithm: String = "SHA-1"): Binary = native("system::HashFunctionValue")

/**
* Computes the HMAC hash and transforms and transforms the binary result into a hexadecimal lower case string.
*/
fun HMACWith(content: Binary, secret: Binary): String =
  lower(
    dw::core::Binaries::toHex(
      HMACBinary(content, secret)
    )
  )

/**
* Computes the MD5 hash and transforms the binary result into a hexadecimal lower case string.
*
* .Transform
* ----
* %dw 2.0
* import dw::Crypto
* output application/json
* ---
* Crypto::MD5("asd" as Binary)
* ----
*
* .Output
* ----
* "7815696ecbf1c96e6894b779456d330e"
* ----
*/
fun MD5(content: Binary): String =
  lower(
    dw::core::Binaries::toHex(content hashWith "MD5")
   )

/**
* Computes the MD5 hash and transforms and transforms the binary result into a hexadecimal lower case string.
*
* .Transform
* ----
* %dw 2.0
* import dw::Crypto
* output application/json
* ---
* Crypto::SHA1("dsasd" as Binary),
* ----
*
* .Output
* ----
* "2fa183839c954e6366c206367c9be5864e4f4a65",
* ----
*/
fun SHA1(content: Binary): String =
  lower(
    dw::core::Binaries::toHex(
      content hashWith "SHA1"
    )
   )
