/**
 * Returns a list of key value pair object describing the object entries
 *
 * .Example
 * [source]
 * ----
 * %dw 2.0
 * import dw::core::Objects
 * ---
 * Objects::entrySet({a: true, b: 1})
 * ----
 *
 * .Output
 * ----
 * [{key: "a", value: true}, {key: "b", value: 1}]
 * ----
 *
 */
fun entrySet<T <: Object>(obj: T) =
  obj pluck (value, key) -> {
    key: key,
    value: value
  }

/**
* Returns the list of key names of an object
*
* .Example
* [source]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* ----
* ["a","b"]
* ----
*/
fun nameSet(obj: Object): Array<String> = obj pluck ($$ as String)

/**
* Returns the list of key values of an object
*
* .Example
* [source]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* ----
* [true,1]
* ----
*/
fun valueSet <K,V>(obj: {K?: V}): Array<V> = obj pluck $

/**
 * Overrides the source with the target object so that the result with contain all the properties from the target
 * plus the properties on the source that are not declared on the target
 *
 * .Example
 * [source]
 * ----
 * %dw 2.0
 * import mergeWith from dw::core::Objects
 * ---
 * {a: true, b: 1} mergeWith {a: false, c: "Test"}
 * ----
 *
 * .Output
 * ----
 * {a: false, b: 1 , c: "Test"}
 * ----
 *
 */
fun mergeWith<T <: Object,V <: Object>(source: T, target: V) =
  (source -- nameSet(target)) ++ target

/**
* Helper method to make `mergetWith` null friendly
*/
fun mergeWith<T <: Object>(a: Null, b: T): T = b

/**
* Helper method to make `mergetWith` null friendly
*/
fun mergeWith<T <: Object>(a: T, b: Null): T = a