/**
* This module contains helper functions to work with Strings.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Strings` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
* Returns the unicode for the first character in an input string.
*
*
* For an empty string, the function fails and returns `Unexpected empty string`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* == Example
*
* This example returns unicode for the "M" in "Mule".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "charCode" : charCode("Mule")
* }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* { "charCode" : 77 }
* ----
*/
fun charCode(text: String): Number = native("system::CharCodeFunctionValue")

/**
* Returns the Unicode for a character at the specified index.
*
*
* This function fails if the index is invalid.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `content` | The input string.
* | `position` | The index (a `Number` type) of a character in the string (as a string array). Note that the index of the first character is `0`.
* |===
*
* == Example
*
* This example returns Unicode for the "u" at index `1` in "MuleSoft".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "charCodeAt" : charCodeAt("MuleSoft", 1)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "charCodeAt": 117 }
* ----
*/
fun charCodeAt(content: String, position: Number): Number =
  charCode(content[position]!)

/**
* Returns a character that matches the input unicode.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `charCode` | The input unicode (a `Number`).
* |===
*
* == Example
*
* This example inputs the unicode number `117` to return the character "u".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "fromCharCode" : fromCharCode(117)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "fromCharCode": "u" }
* ----
*/
fun fromCharCode(charCode: Number): String = native("system::FromCharCodeFunctionValue")

/**
* Pluralizes a singular string.
*
*
* If the input is already plural (for example, "boxes"), the output will match
* the input.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to pluralize.
* |===
*
* == Example
*
* This example pluralizes the input string "box" to return "boxes".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
*  { "pluralize" : pluralize("box") }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "pluralize" : "boxes" }
* ----
**/
fun pluralize(text: String): String = native("system::StringPluralizeOperator")

// Helper function. Do not publish doc.
/**
* Helper function that allows *pluralize* to work with null values.
*/
fun pluralize(text: Null): Null = null

/**
* Converts a plural string to its singular form.
*
*
* If the input is already singular (for example, "box"), the output will match
* the input.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to convert to singular form.
* |===
*
* == Example
*
* This example converts the input string "boxes" to return "box".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* { "singularize" : singularize("boxes") }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "singularize" : "box" }
* ----
**/
fun singularize(text: String): String = native("system::StringSingularizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *singularize* to work with null values.
*/
fun singularize(text: Null): Null = null

/**
* Returns a string in camel case based on underscores in the string.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to convert to camel case.
* |===
*
* == Example
*
* This example converts a string that contains underscores to camel case.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* { "camelize" : camelize("customer_first_name") }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "camelize" : "customerFirstName" }
* ----
**/
fun camelize(text: String): String = native("system::StringCamelizeFunctionValue")

// Helper function. Do not doc.
/**
* Helper function that allows `camelize` to work with null values.
*/
fun camelize(text: Null): Null = null

/**
* Capitalizes the first letter of each word in a string.
*
*
* It also removes underscores between words and puts a space before each
* capitalized word.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to capitalize.
* |===
*
* == Example
*
* This example capitalizes a set of strings.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : capitalize("customer"),
*   "b" : capitalize("customer_first_name"),
*   "c" : capitalize("customer NAME"),
*   "d" : capitalize("customerName")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "Customer",
*   "b": "Customer First Name",
*   "c": "Customer Name",
*   "d": "Customer Name"
* }
* ----
**/
fun capitalize(text: String): String = native("system::StringCapitalizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *capitalize* to work with null values.
*/
fun capitalize(text: Null): Null = null

/**
* Returns a number as an ordinal, such as `1st` or `2nd`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `num` | An input number to return as an ordinal.
* |===
*
* == Example
*
* This example returns a variety of input numbers as ordinals.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : ordinalize(1),
*   "b": ordinalize(2),
*   "c": ordinalize(5),
*   "d": ordinalize(103)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "1st",
*    "b": "2nd",
*    "c": "5th",
*    "d": "103rd"
* }
* ----
**/
fun ordinalize(num: Number): String = native("system::NumberOrdinalizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows `ordinalize` to work with null values.
*/
fun ordinalize(num: Null): Null = null

/**
* Replaces hyphens, spaces, and camel-casing in a string with underscores.
*
*
* If no hyphens, spaces, and camel-casing are present, the output will match
* the input.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* == Example
*
* This example replaces the hyphens and spaces in the input. Notice that
* the input "customer" is not modified in the output.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* {
*    "a" : underscore("customer"),
*    "b" : underscore("customer-first-name"),
*    "c" : underscore("customer NAME"),
*    "d" : underscore("customerName")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "customer",
*    "b": "customer_first_name",
*    "c": "customer_name",
*    "d": "customer_name"
* }
* ----
**/
fun underscore(text: String): String = native("system::StringUnderscoreFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *underscore* to work with null values.
*/
fun underscore(text: Null): Null = null

/**
* Replaces spaces, underscores, and camel-casing in a string with dashes
* (hyphens).
*
*
* If no spaces, underscores, and camel-casing are present, the output will
* match the input.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* == Example
*
* This example replaces the spaces, underscores, and camel-casing in the input.
* Notice that the input "customer" is not modified in the output.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* {
*   "a" : dasherize("customer"),
*   "b" : dasherize("customer_first_name"),
*   "c" : dasherize("customer NAME"),
*   "d" : dasherize("customerName")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "customer",
*   "b": "customer-first-name",
*   "c": "customer-name",
*   "d": "customer-name"
* }
* ----
**/
fun dasherize(text: String): String = native("system::StringDasherizeFunctionValue")

// Helper function. Do not publish doc.
/**
* Helper function that allows *dasherize* to work with null values.
*/
fun dasherize(text: Null): Null = null
