/**
* This is a utility module.
*
* The module is included with Mule runtime. To use it, you must import it to your
* DataWeave code, for example, by adding the line
* `import * from dw::util::Timer` to the header of your script.
*/
%dw 2.0

// A type: TimeMeasurement.
/**
* Represents a start/end time measurement.
*/
type TimeMeasurement<T> =  {start: DateTime, result: T, end: DateTime}

// A type: DurationMeasurement.
/**
* Represents a time taken by a function call.
*/
type DurationMeasurement<T> = {time: Number, result: T}

/**
* Returns the current time in milliseconds.
*
*
* == Example
*
* This example shows the time in milliseconds when the function executed.
*
* === Source
*
* [source,Dataweave, linenums]
* ----
* %dw 2.0
* import * from dw::util::Timer
* output application/json
* ---
* { "currentMilliseconds" : currentMilliseconds() }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* { "currentMilliseconds": 1532923168900 }
* ----
*/
fun currentMilliseconds(): Number =
    toMilliseconds(now())

/**
* Returns the representation of a specified date-time in milliseconds.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `date` | A `DateTime` to evaluate.
* |===
*
* == Example
*
* This example shows a date-time in milliseconds.
*
* === Source
*
* [source,Dataweave, linenums]
* ----
* %dw 2.0
* import * from dw::util::Timer
* output application/json
* ---
* { "toMilliseconds" : toMilliseconds(|2018-07-23T22:03:04.829Z|) }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* { "toMilliseconds": 1532383384829 }
* ----
*/
fun toMilliseconds(date:DateTime): Number =
    date as Number {unit: "milliseconds"}

/**
* Executes the function and returns an object with the taken time in
* milliseconds with the result of the function.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `valueToMeasure` | A function to pass to `duration`.
* |===
*
* == Example
*
* This example passes a `wait` function (defined in the
* header), which returns the result in a `DurationMeasurement`
* object.
*
* === Source
*
* [source,Dataweave, linenums]
* ----
* %dw 2.0
* output application/json
* fun myFunction() = dw::Runtime::wait("My result",100)
* ---
* dw::util::Timer::duration(() -> myFunction())
* ----
*
* === Output
* [source,XML,linenums]
* ----
* {
*   "time": 101,
*   "result": "My result"
* }
* ----
*/
fun duration<T>(valueToMeasure: ()-> T): DurationMeasurement<T> = do {
    var timeResult = time(valueToMeasure)
    ---
    {
        time: toMilliseconds(timeResult.end) - toMilliseconds(timeResult.start),
        result: timeResult.result
    }
}

/**
* Executes the specified function and returns an object with the start time and
* end time with the result of the function.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `valueToMeasure` | A function to pass to `time`.
* |===
*
* == Example
*
* This example passes a `wait` function (defined in the
* header), which returns the result in a `TimeMeasurement`
* object.
*
* [source,Dataweave, linenums]
* ----
* %dw 2.0
* output application/json
* fun myFunction() = dw::Runtime::wait("My result",100)
* ---
* dw::util::Timer::time(() -> myFunction())
* ----
*
* === Output
* [source,XML,linenums]
* ----
* {
*   "start": "2018-09-26T18:58:00.887Z",
*   "result": "My result",
*   "end": "2018-09-26T18:58:00.988Z"
* }
* ----
*/
fun time<T>(valueToMeasure: ()-> T): TimeMeasurement<T> = do {
    var statTime = now()
    var result = valueToMeasure()
    var endTime = now()
    ---
    {
        start: statTime,
        result: result,
        end: endTime
    }

}
