/**
* This module contains helper functions for working with binaries.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Binaries` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Transforms a binary value into a hexadecimal string.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `content` | The `Binary` value to transform.
 * |===
 *
 * === Example
 *
 * This example transforms a binary version of "Mule" (defined in the variable,
 * `myBinary`) to hexadecimal.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * output application/json
 * var myBinary = "Mule" as Binary
 * var testType = typeOf(myBinary)
 * ---
 * {
 *    "binaryToHex" : toHex(myBinary)
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "binaryToHex": "4D756C65" }
 * ----
 */
fun toHex(content: Binary): String = native("system::BinaryToHexFunctionValue")

/**
 * Transforms a hexadecimal string into a binary.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `hexString` | A hexadecimal string to transform.
 * |===
 *
 * === Example
 *
 * This example transforms a hexadecimal string to "Mule".
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * output application/json
 * ---
 * { "hexToBinary": fromHex("4D756C65") }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "hexToBinary": "Mule" }
 * ----
 */
fun fromHex(hexString: String): Binary = native("system::HexToBinaryFunctionValue")

/**
 * Transforms a binary value into a Base64 string.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `content` | The binary value to transform.
 * |===
 *
 * === Example
 *
 * This example transforms a binary to Base64.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * var myBinary = "Mule" as Binary
 * output application/json
 * ---
 * { "BinaryToBase64" : toBase64(myBinary) }
 * ----
 *
 * ==== Output
 * [source,JSON,linenums]
 * ----
 *  { "BinaryToBase64": "TXVsZQ==" }
 * ----
 */
fun toBase64(content: Binary): String = native("system::BinaryToBase64FunctionValue")

/**
 * Transforms a Base64 string into a binary value.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `base64String` | The Base64 string to transform.
 * |===
 *
 * === Example
 *
 * This example takes a Base64 string and transforms it into a binary.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Binaries
 * output application/json
 * ---
 * { "BinaryFromBase64" : fromBase64("TXVsZQ==") }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "BinaryFromBase64": "Mule" }
 * ----
 */
fun fromBase64(base64String: String): Binary = native("system::Base64ToBinaryFunctionValue")

/**
* Splits the specified Binary content into lines and returns the results in an
* array.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `content` | Binary data to read and split.
* | `charset` | String representing the encoding to read.
* |===
*
* === Example
*
* This example transforms binary content, which is separated into new
* lines (`\n`), into a comma-separated array.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Binaries
* var content = read("Line 1\nLine 2\nLine 3\nLine 4\nLine 5\n", "application/octet-stream")
* output application/json
* ---
* {
*    lines : (content readLinesWith "UTF-8"),
*    showType: typeOf(content)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "lines": [ "Line 1", "Line 2", "Line 3", "Line 4", "Line 5" ],
*    "showType": "Binary"
* }
* ----
*/
fun readLinesWith(content: Binary, charset: String): Array<String> = native("system::ReadLinesFunctionValue")

/**
* Writes the specified lines and returns the Binary content.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `content` | Array of items to write.
* | `charset` | String representing the encoding to use when writing.
* |===
*
* === Example
*
* This example inserts a new line (`\n`) after each iteration. Specifically,
* it uses `map` to iterate over the result of `to(1, 10)`, `[1,2,3,4,5]`, then
* writes the specified content ("Line $"), which includes the unnamed variable
* `$` for each number in the array.
*
* Note that without `writeLinesWith  "UTF-8"`, the expression
* `{ lines: to(1, 10) map "Line $" }` simply returns
* an array of line numbers as the value of an object:
* `{ "lines": [ "line 1", "line 2", "line 3", "line 4",  "line 5" ] }`.
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Binaries
* output application/json
* ---
* { lines: to(1, 10) map "Line $" writeLinesWith  "UTF-8" }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "lines": "Line 1\nLine 2\nLine 3\nLine 4\nLine 5\n"
* }
* ----
*/
fun writeLinesWith(content: Array<String>, charset: String): Binary = native("system::WriteLinesFunctionValue")
