/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.impl.util;

import static org.apache.commons.io.IOUtils.closeQuietly;
import static org.apache.commons.io.IOUtils.copy;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

public class ZipUtils {

  /**
   * Unzip the specified archive to the given directory. Equivalent to invoking {@link #unzip(File, File, boolean)} with a
   * {@code true} value for the {@code verify} parameter
   *
   * @param archive   the archive to be unzipped
   * @param directory the target directory
   */
  public static void unzip(File archive, File directory) throws IOException {
    if (directory.exists()) {
      if (!directory.isDirectory()) {
        throw new IOException("Directory is not a directory: " + directory);
      }
    } else {
      if (!directory.mkdirs()) {
        throw new IOException("Could not create directory: " + directory);
      }
    }

    try (ZipFile zip = new ZipFile(archive)) {
      for (Enumeration<? extends ZipEntry> entries = zip.entries(); entries.hasMoreElements();) {
        handleZipEntry(directory, zip, entries.nextElement());
      }
    }
  }

  private static void handleZipEntry(File directory, ZipFile zip, ZipEntry entry) throws IOException, FileNotFoundException {
    File file = verifyZipEntryPath(directory, entry);

    if (entry.isDirectory()) {
      if (!file.exists() && !file.mkdirs()) {
        throw new IOException("Could not create directory: " + file);
      }
    } else {
      if (!file.getParentFile().exists() && !file.getParentFile().mkdirs()) {
        throw new IOException("Unable to create folders for zip entry: " + entry.getName());
      }

      InputStream is = zip.getInputStream(entry);
      OutputStream os = new BufferedOutputStream(new FileOutputStream(file));
      try {
        copy(is, os);
      } finally {
        closeQuietly(is);
        closeQuietly(os);
      }
    }
  }

  private static File verifyZipEntryPath(File targetDirectory, ZipEntry entry) throws IOException {
    Path namePath = Paths.get(entry.getName());
    if (namePath.getRoot() != null) {
      // According to .ZIP File Format Specification (Section 4.4.17), the path can not be absolute
      throw new InvalidZipFileException("Absolute paths are not allowed: " + namePath.toString());
    }

    final File targetFile = new File(targetDirectory, entry.getName());
    // Not specified, but presents a security risk (allows overwriting external files)
    if (!targetFile.getCanonicalPath().startsWith(targetDirectory.getCanonicalPath())) {
      throw new InvalidZipFileException("External paths are not allowed: " + namePath.toString());
    }

    return targetFile;
  }

  private ZipUtils() {
    // nothing to do
  }

  public final static class InvalidZipFileException extends IOException {

    private static final long serialVersionUID = 1L;

    public InvalidZipFileException(String message) {
      super(message);
    }
  }

}
