<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:xs="http://www.w3.org/2001/XMLSchema"
  xmlns:math="http://www.w3.org/2005/xpath-functions/math"
  xmlns:map="http://www.w3.org/2005/xpath-functions/map" xmlns:saxon="http://saxon.sf.net/"
  xmlns:f="MyFunctions" xmlns:svg="http://www.w3.org/2000/svg"
  xmlns:doc="http://www.saxonica.com/ns/doc" exclude-result-prefixes="xs math map saxon svg f doc"
  version="3.0" expand-text="yes">

  <xsl:function name="svg:max.x" as="xs:double"
    doc:doc="find the maximum X for a series of simple pieces">
    <xsl:param name="pieces" as="element()*"/>
    <xsl:sequence select="max($pieces/(@x,@cx,@x1,@x2))"/>
  </xsl:function>
  <xsl:function name="svg:max.y" as="xs:double"
    doc:doc="find the maximum Y for a series of simple pieces">
    <xsl:param name="pieces" as="element()*"/>
    <xsl:sequence select="max($pieces/(@y,@cy,@y1,@y2))"/>
  </xsl:function>
  <xsl:function name="svg:min.x" as="xs:double"
    doc:doc="find the minimum X for a series of simple pieces">
    <xsl:param name="pieces" as="element()*"/>
    <xsl:sequence select="min($pieces/(@x,@cx,@x1,@x2))"/>
  </xsl:function>
  <xsl:function name="svg:min.y" as="xs:double"
    doc:doc="find the minimum Y for a series of simple pieces">
    <xsl:param name="pieces" as="element()*"/>
    <xsl:sequence select="min($pieces/(@y,@cy,@y1,@y2))"/>
  </xsl:function>

  <xsl:mode name="svg:scale" on-no-match="shallow-copy"
    doc:doc="scale and transpose all the positional, rather than size, co-ordinates"/>
  <xsl:template match="@x|@cx|@x1|@x2|@x.max" mode="svg:scale">
    <xsl:param name="x.offset" select="0" as="xs:double" tunnel="yes"/>
    <xsl:param name="x.scale" select="1" as="xs:double" tunnel="yes"/>
    <xsl:attribute name="{local-name()}" select="$x.offset + $x.scale * number(.)"/>
  </xsl:template>
  <xsl:template match="@y|@cy|@y1|@y2|@y.max" mode="svg:scale">
    <xsl:param name="y.offset" select="0" as="xs:double" tunnel="yes"/>
    <xsl:param name="y.scale" select="1" as="xs:double" tunnel="yes"/>
    <xsl:attribute name="{local-name()}" select="$y.offset + $y.scale * number(.)"/>
  </xsl:template>

  <xsl:function name="svg:makeGrid" xmlns="http://www.w3.org/2000/svg" as="element()*">
    <xsl:param name="specs" as="map(*)"/>
    <xsl:param name="width" as="xs:double"/>
    <xsl:param name="height" as="xs:double"/>
    <xsl:sequence select="svg:makeGrid.horizontal($specs,$width,$height)"/>
    <xsl:sequence select="svg:makeGrid.vertical($specs,$width,$height)"/>
  </xsl:function>
  <xsl:function name="svg:makeGrid.horizontal" xmlns="http://www.w3.org/2000/svg" as="element()*">
    <xsl:param name="specs" as="map(*)"/>
    <xsl:param name="width" as="xs:double"/>
    <xsl:param name="height" as="xs:double"/>
    <xsl:variable name="format.y" select="($specs('format.y'),format-number(?,'0'))[1]"/>
    <g>
      <xsl:for-each select="0 to $specs('n.units.y')">
        <xsl:variable name="m5" select=". mod 5 = 0"/>
        <xsl:variable name="y" select="$height - $specs('scale.y') * $specs('unit.y') * ."/>
        <line stroke="grey" stroke-width="{if($m5) then 1 else 0.5}" x1="0" x2="{$width}" y1="{$y}"
          y2="{$y}"/>
        <xsl:if test="$m5">
          <text class="axis" font-size="10" text-anchor="end" x="0" y="{$y}"
            >{string($specs('axis.y.format')($specs('unit.y') * .))}</text>
        </xsl:if>
      </xsl:for-each>
      <text transform="rotate(-90,-10,{$height idiv 2})" y="{$height idiv 2}" x="-10"
        class="axis label" text-anchor="middle">{$specs('axis.y.label')}</text>
    </g>
  </xsl:function>
  <xsl:function name="svg:makeGrid.vertical" xmlns="http://www.w3.org/2000/svg" as="element()*">
    <xsl:param name="specs" as="map(*)"/>
    <xsl:param name="width" as="xs:double"/>
    <xsl:param name="height" as="xs:double"/>
    <g>
      <xsl:for-each select="0 to $specs('n.units.x')">
        <xsl:variable name="m5" select=". mod 5 = 0"/>
        <xsl:variable name="x" select="$specs('scale.x') * $specs('unit.x') * ."/>
        <line stroke="grey" stroke-width="{if($m5) then 1 else 0.5}" y1="0" y2="{$height}" x1="{$x}"
          x2="{$x}"/>
        <xsl:if test="$m5">
          <text class="axis value" text-anchor="middle" font-size="10" y="{$height + 10}" x="{$x}"
            >{$specs('axis.x.format')($specs('unit.x') * .)}</text>
        </xsl:if>
      </xsl:for-each>
      <text x="{$width idiv 2}" y="{$height + 30}" class="axis label" text-anchor="middle"
        >{$specs('axis.x.label')}</text>
    </g>
  </xsl:function>

  <xsl:function name="svg:axis-minor-tick" as="xs:double">
    <xsl:param name="min" as="xs:double"/>
    <xsl:param name="max" as="xs:double"/>
    <xsl:variable name="upper" select="10 * ceiling($max div 10)"/>
    <xsl:variable name="lower" select="10 * floor($min div 10)"/>
    <xsl:variable name="range" select="$upper - $lower"/>
    <xsl:variable name="increment"
      select="math:pow(10.0,floor(math:log($range div 20) div math:log(10.0)))"/>
    <xsl:sequence select="if($range div $increment gt 50) then 10 * $increment else $increment"/>
  </xsl:function>
  <xsl:function name="svg:axis-ticks" as="xs:double*">
    <xsl:param name="min" as="xs:double"/>
    <xsl:param name="max" as="xs:double"/>
    <xsl:variable name="upper" select="10 * ceiling($max div 10)"/>
    <xsl:variable name="lower" select="10 * floor($min div 10)"/>
    <xsl:variable name="range" select="$upper - $lower"/>
    <xsl:variable name="increment" select="svg:axis-minor-tick($min,$max)"/>
    <xsl:sequence
      select="for $i in 0 to xs:integer($range idiv $increment) return
      for $v in ($lower + $i * $increment) return if($v ge $min and $v le $max) then $v else ()"
    />
  </xsl:function>

  <xsl:function name="svg:pie" as="element()*" xmlns="http://www.w3.org/2000/svg" doc:doc="Make a pie chart">
    <xsl:param name="r" as="xs:double"/>
    <xsl:param name="x" as="xs:double"/>
    <xsl:param name="y" as="xs:double"/>
    <xsl:param name="values" as="map(*)*" doc:doc="The elements for the wedges - 
      each map having entries: 'val','class'?, 'label'?"/>
    <xsl:variable name="scale" select="360.0 div sum($values!.('val'))"/>
    <g transform="translate({$x},{$y})">
      <xsl:variable name="wedges"
        select="tail(fold-left(
      $values,
      (0),
      function($so-far,$next) { 
        let $off := head($so-far), 
        $val := $next('val'), 
        $scaled := $scale * $val,
        $subsequent := $scaled + $off  return
        ($subsequent,tail($so-far),svg:wedge($off,$subsequent,$r))
      }))"/>
      <xsl:for-each select="$wedges">
        <xsl:variable name="m" select="subsequence($values,position(),1)" as="map(*)"/>
        <xsl:copy>
          <xsl:attribute name="class" select="'pie',$m('class')"/>
          <xsl:sequence select="@points,*"/>
        </xsl:copy>
      </xsl:for-each>
      <xsl:for-each select="$wedges">
        <xsl:variable name="m" select="subsequence($values,position(),1)" as="map(*)"/>
        <xsl:if test="$m('label')">
          <text class="pie label" x="{@m.x}" y="{@m.y}"
            text-anchor="{if(number(@m.x) lt 0) then 'end' else 'start'}">{$m('label')}</text>
        </xsl:if>
      </xsl:for-each>
    </g>
  </xsl:function>

  <xsl:function name="svg:wedge" as="element()*" xmlns="http://www.w3.org/2000/svg">
    <xsl:param name="theta1" as="xs:double" doc:doc="in degrees"/>
    <xsl:param name="theta2" as="xs:double" doc:doc="in degrees"/>
    <xsl:param name="r" as="xs:double"/>
    <xsl:variable name="bits"
      select="
      $theta1,
      for $i in 1 to ($theta2 - $theta1) idiv 10 return $theta1 + $i * 10,
      $theta2"/>
    <xsl:variable name="mid" select="svg:rTheta(($theta1 + $theta2) div 2,1.1 * $r)"/>
    <polygon m.x="{$mid[1]}" m.y="{$mid[2]}" points="0 0 {$bits!svg:rTheta(.,$r)}"/>
  </xsl:function>
  <xsl:function name="svg:rTheta" as="xs:double*">
    <xsl:param name="theta" as="xs:double" doc:doc="in degrees"/>
    <xsl:param name="r" as="xs:double"/>
    <xsl:sequence
      select="let $a := ($theta * math:pi() div 180.0)
      return ($r *  math:cos($a),$r * math:sin($a))"
    />
  </xsl:function>
</xsl:stylesheet>
