/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.oauth2.internal.clientcredentials;

import org.mule.DefaultMuleEvent;
import org.mule.DefaultMuleMessage;
import org.mule.MessageExchangePattern;
import org.mule.api.DefaultMuleException;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.construct.Flow;
import org.mule.module.http.api.HttpHeaders;
import org.mule.module.oauth2.internal.AbstractTokenRequestHandler;
import org.mule.module.oauth2.internal.ApplicationCredentials;
import org.mule.module.oauth2.internal.MuleEventLogger;
import org.mule.module.oauth2.internal.OAuthConstants;
import org.mule.module.oauth2.internal.OAuthTokenMuleException;
import org.mule.module.oauth2.internal.TokenNotFoundException;
import org.mule.module.oauth2.internal.TokenResponseProcessor;
import org.mule.module.oauth2.internal.authorizationcode.TokenResponseConfiguration;
import org.mule.module.oauth2.internal.authorizationcode.state.ResourceOwnerOAuthContext;
import org.mule.module.oauth2.internal.tokenmanager.TokenManagerConfig;
import org.mule.transport.NullPayload;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;

import org.apache.commons.codec.binary.Base64;

/**
 * Handler for calling the token url, parsing the response and storing the oauth context data.
 */
public class ClientCredentialsTokenRequestHandler extends AbstractTokenRequestHandler implements Initialisable, Disposable
{

    private String scopes;
    private ApplicationCredentials applicationCredentials;
    private TokenResponseConfiguration tokenResponseConfiguration = new TokenResponseConfiguration();
    private TokenManagerConfig tokenManager;
    private boolean encodeClientCredentialsInBody = false;
    private Lock refreshLock;
    private MuleEventLogger muleEventLogger = new MuleEventLogger(logger);

    public void setApplicationCredentials(ApplicationCredentials applicationCredentials)
    {
        this.applicationCredentials = applicationCredentials;
    }

    public void setScopes(String scopes)
    {
        this.scopes = scopes;
    }

    public void setTokenResponseConfiguration(TokenResponseConfiguration tokenResponseConfiguration)
    {
        this.tokenResponseConfiguration = tokenResponseConfiguration;
    }

    private void setMapPayloadWithTokenRequestParameters(final MuleEvent event) throws MuleException
    {
        final Map<String, String> formData = new HashMap<String, String>();
        formData.put(OAuthConstants.GRANT_TYPE_PARAMETER, OAuthConstants.GRANT_TYPE_CLIENT_CREDENTIALS);
        String clientId = applicationCredentials.getClientId();
        String clientSecret = applicationCredentials.getClientSecret();
        if (encodeClientCredentialsInBody)
        {
            formData.put(OAuthConstants.CLIENT_ID_PARAMETER, clientId);
            formData.put(OAuthConstants.CLIENT_SECRET_PARAMETER, clientSecret);
        }
        else
        {
            String encodedCredentials = Base64.encodeBase64String(String.format("%s:%s", clientId, clientSecret).getBytes());
            event.getMessage().setOutboundProperty(HttpHeaders.Names.AUTHORIZATION, "Basic " + encodedCredentials);
        }
        if (scopes != null)
        {
            formData.put(OAuthConstants.SCOPE_PARAMETER, scopes);
        }
        event.getMessage().setPayload(formData);
    }

    public void refreshAccessToken() throws MuleException
    {
        final boolean lockWasAcquired = refreshLock.tryLock();
        try
        {
            if (lockWasAcquired)
            {
                doRefreshAccessToken();
            }
        }
        finally
        {
            if (lockWasAcquired)
            {
                refreshLock.unlock();
            }
        }
        if (!lockWasAcquired)
        {
            //if we couldn't acquire the lock then we wait until the other thread updates the token.
            waitUntilLockGetsReleased();
        }
    }

    private void waitUntilLockGetsReleased()
    {
        refreshLock.lock();
        refreshLock.unlock();
    }

    public void doRefreshAccessToken() throws MuleException
    {
        try
        {
            final DefaultMuleEvent accessTokenEvent = new DefaultMuleEvent(new DefaultMuleMessage(NullPayload.getInstance(), getMuleContext()), MessageExchangePattern.REQUEST_RESPONSE, new Flow("test", getMuleContext()));
            setMapPayloadWithTokenRequestParameters(accessTokenEvent);
            final MuleEvent response;
            response = invokeTokenUrl(accessTokenEvent);
            final TokenResponseProcessor tokenResponseProcessor = TokenResponseProcessor.createClientCredentialsProcessor(tokenResponseConfiguration, getMuleContext().getExpressionManager());
            tokenResponseProcessor.process(response);

            if (logger.isDebugEnabled())
            {
                logger.debug("Retrieved access token, refresh token and expires from token url are: %s, %s, %s",
                             tokenResponseProcessor.getAccessToken(),
                             tokenResponseProcessor.getRefreshToken(),
                             tokenResponseProcessor.getExpiresIn());
            }

            if (!tokenResponseContentIsValid(tokenResponseProcessor))
            {
                throw new TokenNotFoundException(response, tokenResponseProcessor);
            }

            final ResourceOwnerOAuthContext defaultUserState = tokenManager.getConfigOAuthContext().getContextForResourceOwner(ResourceOwnerOAuthContext.DEFAULT_RESOURCE_OWNER_ID);
            defaultUserState.setAccessToken(tokenResponseProcessor.getAccessToken());
            defaultUserState.setExpiresIn(tokenResponseProcessor.getExpiresIn());
            final Map<String,Object> customResponseParameters = tokenResponseProcessor.getCustomResponseParameters();
            for (String paramName : customResponseParameters.keySet())
            {
                defaultUserState.getTokenResponseParameters().put(paramName, customResponseParameters.get(paramName));
            }
            tokenManager.getConfigOAuthContext().updateResourceOwnerOAuthContext(defaultUserState);
        }
        catch (TokenNotFoundException e)
        {
            logger.error(String.format("Could not extract access token or refresh token from token URL. Access token is %s, Refresh token is %s",
                                       e.getTokenResponseProcessor().getAccessToken(), e.getTokenResponseProcessor().getRefreshToken()));
            muleEventLogger.logContent(e.getTokenUrlResponse());
            throw new OAuthTokenMuleException(e);
        }
        catch (TokenUrlResponseException e)
        {
            logger.error((String.format("HTTP response from token URL %s returned a failure status code", getTokenUrl())));
            muleEventLogger.logContent(e.getTokenUrlResponse());
            throw new OAuthTokenMuleException(e);
        }
    }

    private boolean tokenResponseContentIsValid(TokenResponseProcessor tokenResponseProcessor)
    {
        return tokenResponseProcessor.getAccessToken() != null;
    }

    public void setTokenManager(TokenManagerConfig tokenManager)
    {
        this.tokenManager = tokenManager;
    }

    public void setEncodeClientCredentialsInBody(boolean encodeClientCredentialsInBody)
    {
        this.encodeClientCredentialsInBody = encodeClientCredentialsInBody;
    }

    @Override
    public void initialise() throws InitialisationException
    {
        refreshLock = getMuleContext().getLockFactory().createLock(ClientCredentialsGrantType.class.getSimpleName() + "#refreshAccessToken_" + hashCode());
    }
}
