/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.mule.testing.processing.strategies.test.api;

import static java.lang.Thread.currentThread;
import static org.slf4j.LoggerFactory.getLogger;

import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.List;

/**
 * When a thread jump happens, the ThreadLocals are cleared.
 * So, this object keeps a ThreadLocal with an arbitrary Integer:
 * - If the ThreadLocal is present, it adds the current processor to a temporary list (there weren't any thread jump).
 * - If the ThreadLocal isn't present, it saves a List with only the current processor, and moves the previous List
 * to another data structure. The Lists in that data structure will have the processors that were executed without a
 * thread jump.
 *
 * The user can track the execution and completion of an operation, and ask for the "phase" when it was executed or
 * completed. If the "phase" is the same for two events, it means that there weren't any thread jumps between them.
 * If the "phase" changes between two events, it's because there were a thread jump.
 */
public class ExecutionThreadTracker {

    private final List<List<String>> processorsThatWereExecutedWithoutAThreadJump = new ArrayList<>();

    private static final Integer SENTINEL_VALUE = 52;
    private ThreadLocal<Integer> sentinelThreadLocal = null;
    private List<String> processorsSinceLastThreadJump;

    private static final String EXECUTION_PREFIX = "Execution";
    private static final String COMPLETION_PREFIX = "Completion";

    private static final Logger LOGGER = getLogger(ExecutionThreadTracker.class);

    public void trackExecutionThread(String key) {
        LOGGER.info("Key [{}] executed in thread: [{}]", key, currentThread().getName());
        resetSentinelIfNeeded();
        addToCurrentPhase(execution(key));
    }

    public void trackCompletionThread(String key) {
        LOGGER.info("Key [{}] completed in thread: [{}]", key, currentThread().getName());
        resetSentinelIfNeeded();
        addToCurrentPhase(completion(key));
    }

    public Integer getExecutionThreadPhase(String key) {
        return search(execution(key));
    }

    public Integer getCompletionThreadPhase(String key) {
        return search(completion(key));
    }

    private void addToCurrentPhase(String decoratedKey) {
        processorsSinceLastThreadJump.add(decoratedKey);
    }

    private Integer search(String decoratedKey) {
        int phaseOrder = 0;
        for (List<String> phase : processorsThatWereExecutedWithoutAThreadJump) {
            if (phase.contains(decoratedKey)) {
                return phaseOrder;
            }
            phaseOrder += 1;
        }
        return null;
    }

    private void resetSentinelIfNeeded() {
        if (comingFromAThreadJump()) {
            setSentinel();
            saveProcessorsSincePreviousThreadJump();
            resetProcessorsList();
        }
    }

    private void resetProcessorsList() {
        processorsSinceLastThreadJump = new ArrayList<>();
    }

    private void saveProcessorsSincePreviousThreadJump() {
        if (processorsSinceLastThreadJump != null && !processorsSinceLastThreadJump.isEmpty()) {
            processorsThatWereExecutedWithoutAThreadJump.add(processorsSinceLastThreadJump);
        }
    }

    private boolean comingFromAThreadJump() {
        return sentinelThreadLocal == null || sentinelThreadLocal.get() == null;
    }

    private void setSentinel() {
        sentinelThreadLocal = new ThreadLocal<>();
        sentinelThreadLocal.set(SENTINEL_VALUE);
    }

    private static String execution(String key) {
        return EXECUTION_PREFIX + "(" + key + ")";
    }

    private static String completion(String key) {
        return COMPLETION_PREFIX + "(" + key + ")";
    }
}
