/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package cookBook.server;

import cookBook.server.authentication.BasicAuthHandler;
import cookBook.server.authentication.BearerAuthHandler;
import cookBook.server.authentication.CheckUserIsAuthenticatedHandler;
import cookBook.server.authentication.OAuth2AuthorizationCodeHandler;
import cookBook.server.authentication.OAuth2ClientCredentialsHandler;
import cookBook.server.operation.GetAllRecipesHypermediaBodyPagination;
import cookBook.server.operation.GetAllRecipesHypermediaHeaderPagination;
import cookBook.server.operation.GetAllRecipesMarkerPagination;
import cookBook.server.operation.GetAllRecipesOffsetPagination;
import cookBook.server.operation.GetAllRecipesPageNumberPagination;
import cookBook.server.operation.GetIngredientById;
import cookBook.server.operation.GetInvalidRecipesByCategoryId;
import cookBook.server.operation.GetRecipeById;
import cookBook.server.operation.HealthCheck;
import cookBook.server.operation.PostCategory;
import cookBook.server.operation.PostIngredient;
import cookBook.server.operation.PostInvalidCategory;
import cookBook.server.operation.PostRecipe;
import io.vertx.core.AbstractVerticle;
import io.vertx.core.Vertx;
import io.vertx.ext.web.Router;
import io.vertx.ext.web.handler.BodyHandler;
import io.vertx.ext.web.handler.LoggerHandler;

public class CookBookMockServer extends AbstractVerticle {

  public static CookBookMockServer server;

  private final Vertx vertx = Vertx.vertx();
  private final Integer port;
  private final Integer redirectPort;

  public static CookBookMockServer getInstance(Integer port, Integer redirectPort) {
    if (server == null) {
      server = new CookBookMockServer(port, redirectPort);
    }
    return server;
  }

  public CookBookMockServer(Integer port, Integer redirectPort) {
    this.port = port;
    this.redirectPort = redirectPort;
  }

  public void stopServer() {
    vertx.close();
  }

  @Override
  public void start() {
    Router router = Router.router(vertx);

    router
        .route()
        .handler(LoggerHandler.create())
        .handler(BodyHandler.create())
        .blockingHandler(BodyHandler.create())
        .handler(new BasicAuthHandler())
        .handler(new OAuth2AuthorizationCodeHandler(redirectPort))
        .handler(new OAuth2ClientCredentialsHandler())
        .handler(new BearerAuthHandler())
        .handler(new CheckUserIsAuthenticatedHandler());

    router.get("/ingredient/:id").blockingHandler(new GetIngredientById());
    router.post("/ingredient").blockingHandler(new PostIngredient());
    router.get("/recipe/:id").blockingHandler(new GetRecipeById());
    router.post("/recipe").blockingHandler(new PostRecipe());
    router.get("/category/:id/recipes").blockingHandler(new GetInvalidRecipesByCategoryId());
    router.post("/category/invalid").blockingHandler(new PostInvalidCategory());
    router.get("/health").blockingHandler(new HealthCheck());
    router.post("/category").blockingHandler(new PostCategory());
    router.get("/recipes-page-number").blockingHandler(new GetAllRecipesPageNumberPagination());
    router.get("/recipes-offset").blockingHandler(new GetAllRecipesOffsetPagination());
    router.get("/recipes-marker").blockingHandler(new GetAllRecipesMarkerPagination());
    router.get("/recipes-hypermedia-body").blockingHandler(new GetAllRecipesHypermediaBodyPagination());
    router.get("/recipes-hypermedia-header").blockingHandler(new GetAllRecipesHypermediaHeaderPagination());

    vertx.createHttpServer()
        .requestHandler(router)
        .listen(port, result -> {
          if (!result.succeeded()) {
            result.cause().printStackTrace();
          }
        });
  }

}
