/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package cookBook.server.authentication;

import static cookBook.server.authentication.CheckUserIsAuthenticatedHandler.IS_AUTHORIZED_VAR;
import static cookBook.server.authentication.RequestAssertionUtils.assertBodyHasExpectedValue;
import static cookBook.server.authentication.RequestAssertionUtils.assertQueryParamHasExpectedValue;
import static io.vertx.core.http.HttpHeaders.AUTHORIZATION;
import static io.vertx.core.http.HttpHeaders.CONTENT_TYPE;
import static io.vertx.core.logging.LoggerFactory.getLogger;
import static io.vertx.ext.web.client.WebClient.create;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import io.vertx.core.Handler;
import io.vertx.core.logging.Logger;
import io.vertx.ext.web.RoutingContext;
import io.vertx.ext.web.client.WebClient;

public class OAuth2AuthorizationCodeHandler implements Handler<RoutingContext> {

  private final static Logger LOGGER = getLogger(OAuth2AuthorizationCodeHandler.class);

  public final static String EXPECTED_AUTHORIZATION_URL = "/authorization-code-authorize";
  public final static String EXPECTED_ACCESS_TOKEN_URL = "/authorization-code-access-token";

  public final static String CLIENT_ID_PARAM_NAME = "client_id";
  public final static String CLIENT_SECRET_PARAM_NAME = "client_secret";
  public final static String RESPONSE_TYPE_PARAM_NAME = "response_type";
  public final static String SCOPE_PARAM_NAME = "scope";
  public final static String REDIRECT_URI_PARAM_NAME = "redirect_uri";
  public final static String CODE_PARAM_NAME = "code";

  public final static String EXPECTED_CLIENT_ID = "my_client_id";
  public final static String EXPECTED_CLIENT_SECRET = "my_client_secret";
  public final static String EXPECTED_SCOPES = "READ";
  public final static String EXPECTED_RESPONSE_TYPE = "code";

  public final static String EXPECTED_REDIRECT_URI = "http://0.0.0.0:%s/mule-callback";

  public final static String EXPECTED_CODE = "my-authorization-code";
  public final static String EXPECTED_TOKEN = "my-authorization-token-via-authorization-code";

  public final static String TOKEN_RESPONSE = "{\n"
      + "    \"expires_in\":86400,\n"
      + "    \"token_type\":\"bearer\",\n"
      + "    \"refresh_token\":\"" + EXPECTED_TOKEN + "\",\n"
      + "    \"access_token\":\"" + EXPECTED_TOKEN + "\"\n"
      + "}";

  private final static int SC_UNAUTHORIZED = 401;

  private final Integer redirectPort;

  public OAuth2AuthorizationCodeHandler(Integer redirectPort) {
    this.redirectPort = redirectPort;
  }

  @Override
  public void handle(RoutingContext context) {
    switch (context.request().path().toLowerCase()) {
      case EXPECTED_AUTHORIZATION_URL:
        authorize(context);
        break;
      case EXPECTED_ACCESS_TOKEN_URL:
        createToken(context);
        break;
      default:
        validateToken(context);
    }
  }

  private void validateToken(RoutingContext context) {
    String authorizationHeader = context.request().getHeader(AUTHORIZATION);

    if (isNotBlank(authorizationHeader) && authorizationHeader.equals("Bearer " + EXPECTED_TOKEN)) {
      context.data().put(IS_AUTHORIZED_VAR, true);
    }
    context.next();
  }

  private void createToken(RoutingContext context) {
    boolean correctRequest =
        assertBodyHasExpectedValue(context, CODE_PARAM_NAME, EXPECTED_CODE) &&
            assertBodyHasExpectedValue(context, CLIENT_ID_PARAM_NAME, EXPECTED_CLIENT_ID) &&
            assertBodyHasExpectedValue(context, CLIENT_SECRET_PARAM_NAME, EXPECTED_CLIENT_SECRET) &&
            assertBodyHasExpectedValue(context, REDIRECT_URI_PARAM_NAME, String.format(EXPECTED_REDIRECT_URI, redirectPort));

    if (correctRequest) {
      context.response().setStatusCode(200).putHeader(CONTENT_TYPE, "application/json").end(TOKEN_RESPONSE);
    } else {
      context.response().setStatusCode(SC_UNAUTHORIZED).end();
    }
  }

  private void authorize(RoutingContext context) {
    boolean correctRequest =
        assertQueryParamHasExpectedValue(context, RESPONSE_TYPE_PARAM_NAME, EXPECTED_RESPONSE_TYPE) &&
            assertQueryParamHasExpectedValue(context, CLIENT_ID_PARAM_NAME, EXPECTED_CLIENT_ID) &&
            assertQueryParamHasExpectedValue(context, SCOPE_PARAM_NAME, EXPECTED_SCOPES) &&
            assertQueryParamHasExpectedValue(context, REDIRECT_URI_PARAM_NAME,
                                             String.format(EXPECTED_REDIRECT_URI, redirectPort));

    if (correctRequest) {
      sendAuthorizeResponse(context);
      context.response().setStatusCode(200).end();
    } else {
      context.response().setStatusCode(SC_UNAUTHORIZED).end();
    }
  }

  private void sendAuthorizeResponse(RoutingContext context) {
    try {
      WebClient client = create(context.vertx());

      client
          .get(redirectPort, "127.0.0.1", "/mule-callback")
          .addQueryParam(CODE_PARAM_NAME, EXPECTED_CODE)
          .send(ar -> {
            if (ar.succeeded()) {
              LOGGER.info("Received response with status code " + ar.result().statusCode());
            } else {
              LOGGER.error("Something went wrong " + ar.cause().getMessage());
            }
          });
    } catch (Exception e) {
      LOGGER.error("Error sending authorize response", e);
      throw e;
    }
  }

}
