/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package cookBook.server.authentication;

import static cookBook.server.authentication.AuthenticationUtils.validateBasicAuthorizationHeader;
import static cookBook.server.authentication.CheckUserIsAuthenticatedHandler.IS_AUTHORIZED_VAR;
import static cookBook.server.authentication.RequestAssertionUtils.assertBodyHasExpectedValue;
import static io.vertx.core.http.HttpHeaders.AUTHORIZATION;
import static io.vertx.core.http.HttpHeaders.CONTENT_TYPE;

import io.vertx.core.Handler;
import io.vertx.ext.web.RoutingContext;
import org.apache.commons.lang3.StringUtils;

public class OAuth2ClientCredentialsHandler implements Handler<RoutingContext> {

  public final static String EXPECTED_ACCESS_TOKEN_URL = "/client-credentials-access-token";

  public final static String GRANT_TYPE_PARAM_NAME = "grant_type";

  public final static String EXPECTED_CLIENT_ID = "my_client_id";
  public final static String EXPECTED_CLIENT_SECRET = "my_client_secret";
  public final static String EXPECTED_GRANT_TYPE = "client_credentials";

  public final static String EXPECTED_TOKEN = "my-authorization-token-via-client-credentials";

  public final static String TOKEN_RESPONSE = "{\n"
      + "    \"expires_in\":86400,\n"
      + "    \"token_type\":\"bearer\",\n"
      + "    \"refresh_token\":\"" + EXPECTED_TOKEN + "\",\n"
      + "    \"access_token\":\"" + EXPECTED_TOKEN + "\"\n"
      + "}";

  private final static int SC_UNAUTHORIZED = 401;

  @Override
  public void handle(RoutingContext context) {
    if (EXPECTED_ACCESS_TOKEN_URL.equalsIgnoreCase(context.request().path())) {
      createToken(context);
    } else {
      validateToken(context);
    }
  }

  private void validateToken(RoutingContext context) {
    String authorizationHeader = context.request().getHeader(AUTHORIZATION);

    if (StringUtils.isNotBlank(authorizationHeader) && authorizationHeader.equals("Bearer " + EXPECTED_TOKEN)) {
      context.data().put(IS_AUTHORIZED_VAR, true);
    }
    context.next();
  }

  private void createToken(RoutingContext context) {
    boolean correctRequest =
        assertBodyHasExpectedValue(context, GRANT_TYPE_PARAM_NAME, EXPECTED_GRANT_TYPE)
            && validateBasicAuthorizationHeader(context.request(), EXPECTED_CLIENT_ID, EXPECTED_CLIENT_SECRET);

    if (correctRequest) {
      context.response().setStatusCode(200).putHeader(CONTENT_TYPE, "application/json").end(TOKEN_RESPONSE);
    } else {
      context.response().setStatusCode(SC_UNAUTHORIZED).end();
    }
  }

}
