/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package cookBook.server.authentication;

import static io.vertx.core.logging.LoggerFactory.getLogger;
import static java.lang.String.format;
import static java.net.URLEncoder.encode;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.io.UnsupportedEncodingException;
import java.util.List;

import io.vertx.core.logging.Logger;
import io.vertx.ext.web.RoutingContext;

public class RequestAssertionUtils {

  private static final Logger LOGGER = getLogger(RequestAssertionUtils.class);

  public static boolean assertQueryParamHasExpectedValue(RoutingContext context, String paramName, String expectedValue) {
    if (context == null) {
      LOGGER.error("RoutingContext is null");
      return false;
    }

    List<String> queryParams = context.queryParam(paramName);

    if (queryParams.isEmpty()) {
      LOGGER.error(format("Query parameter '%s' not found in request.", paramName));
      return false;
    }

    if (!queryParams.get(0).equals(expectedValue)) {
      LOGGER.error(format("Query parameter '%s expected value (%s) mismatch. Actual value: %s",
                          paramName, expectedValue, queryParams.get(0)));
      return false;
    }

    return true;
  }

  public static boolean assertHeaderHasExpectedValue(RoutingContext context, String headerName, String expectedValue) {
    if (context == null) {
      LOGGER.error("RoutingContext is null");
      return false;
    }

    if (context.request() == null) {
      LOGGER.error("Request is null");
      return false;
    }

    String headerValue = context.request().getHeader(headerName);

    if (isBlank(headerValue)) {
      LOGGER.error(format("Header '%s' not found in request.", headerName));
      return false;
    }

    if (!headerValue.equals(expectedValue)) {
      LOGGER.error(format("Header '%s' expected value (%s) mismatch. Actual value: %s",
                          headerName, expectedValue, headerValue));
      return false;
    }

    return true;
  }

  public static boolean headerIsPresent(RoutingContext context, String headerName) {
    if (context == null) {
      LOGGER.error("RoutingContext is null");
      return false;
    }

    if (context.request() == null) {
      LOGGER.error("Request is null");
      return false;
    }

    String headerValue = context.request().getHeader(headerName);

    return isNotBlank(headerValue);
  }

  public static boolean assertBodyHasExpectedValue(RoutingContext context, String paramName, String expectedValue) {
    if (context == null) {
      LOGGER.error("RoutingContext is null");
      return false;
    }

    String body = context.getBodyAsString();

    try {
      if (body.contains(paramName + "=" + encode(expectedValue, UTF_8.name()))) {
        return true;
      } else {
        LOGGER.error(format("Parameter '%s' with expected value (%s) not found in request body. \n Actual Body: %s",
                            paramName, expectedValue, body));
        return false;
      }
    } catch (UnsupportedEncodingException e) {
      LOGGER.error(e.getMessage(), e);
      return false;
    }
  }

}
