package com.mulesoft.connectors.cookBook.internal.operation;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.error.RestErrorTypeProvider;
import com.mulesoft.connectivity.rest.commons.api.operation.BaseRestOperation;
import com.mulesoft.connectors.cookBook.internal.metadata.CategoryParameterAddInvalidCategoryOperationInputResolver;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.metadata.TypeResolver;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.InputStream;
import java.util.Optional;
import java.util.function.Function;

import static java.util.Optional.of;
import static org.mule.runtime.api.metadata.DataType.fromType;
import static org.mule.sdk.api.annotation.param.MediaType.APPLICATION_JSON;

public class AddInvalidCategoryOperation extends BaseRestOperation {

  @Override
  protected Function<HttpResponse, Optional<ErrorTypeDefinition<? extends Enum<?>>>> httpResponseToErrorTypeDefinition() {
    return httpResponse -> {
      // Status code = 200
      if (httpResponse.getStatusCode() >= 200 && httpResponse.getStatusCode() < 300) {
        // Getting the error property from payload
        Boolean isResponseWithError = (Boolean) getExpressionLanguage()
            .evaluate("#[payload.error]",
                      fromType(Boolean.class),
                      BindingContext.builder()
                          .addBinding("payload", new TypedValue(httpResponse.getEntity().getContent(), DataType.JSON_STRING))
                          .build())
            .getValue();

        // If error property is true, return connectivity error, at the end, a custom module exception
        // could be returned
        if (isResponseWithError) {
          return of(RestError.CONNECTIVITY);
        } else {
          return Optional.empty();
        }
      }
      // Default behavior, it will be catched as a HTTP error
      return super.httpResponseToErrorTypeDefinition().apply(httpResponse);
    };
  }

  @Override
  protected <E extends Enum<E>> ModuleException createModuleException(HttpResponse httpResponse,
                                                                      ErrorTypeDefinition<E> errorTypeDefinition) {

    // Here a connectivity error was thrown before but the description field is needed in the response
    String description = (String) getExpressionLanguage()
        .evaluate("#[payload.description]",
                  DataType.TEXT_STRING,
                  BindingContext.builder()
                      .addBinding("payload", new TypedValue(httpResponse.getEntity().getContent(), DataType.JSON_STRING))
                      .build())
        .getValue().toString();

    return new ModuleException(description, errorTypeDefinition);
  }

  @MediaType(APPLICATION_JSON)
  // Default mule error types
  @Throws(RestErrorTypeProvider.class)
  public void addInvalidCategory(
                                 @Connection RestConnection connection,
                                 CompletionCallback<InputStream, Object> completionCallback,
                                 @TypeResolver(CategoryParameterAddInvalidCategoryOperationInputResolver.class) InputStream category) {
    try {
      // Build the endpoint full uri. Example http://devkit-cookbook.cloudhub.io/rest/category
      String fullUri = connection.getBaseUri() + "/category/invalid";

      // Build the http request using HttpRequestBuilder
      HttpRequest request = HttpRequest.builder()
          .uri(fullUri)
          .entity(new InputStreamHttpEntity(category))
          .method(HttpConstants.Method.POST)
          .build();

      // Do sendAsync
      connection.sendAsync(request)
          // This is necessary to make the httpResponse.getEntity().getContent() repeatable
          // And make it repeatable is necessary because at least two reads are needed
          // The first one to get de signal property and the second one to get the response message
          // Without this, with the first read, the httpResponse.getEntity().getContent() InputStream
          // will be read and the following needed read will find an empty
          .thenApply(httpResponse -> {
            httpResponse = HttpResponse.builder()
                .headers(httpResponse.getHeaders())
                .statusCode(httpResponse.getStatusCode())
                .reasonPhrase(httpResponse.getReasonPhrase())
                .entity(new ByteArrayHttpEntity(IOUtils.toByteArray(httpResponse.getEntity().getContent())))
                .build();
            return httpResponse;
          })
          // If http status code error present, throw error. Example, the SaaS returns a status code 404
          // but a override method for createModuleException is added to change the response message
          .thenApply(throwModuleExceptionIfErrorResponse())
          // Set the callback in success with a Result object with the whole content returned by SaaS
          .thenAccept(httpResponse -> completionCallback.success(Result
              .<InputStream, Object>builder()
              .output(httpResponse.getEntity().getContent())
              .build()))
          // Set the callback in error if the sendAsync CompletableFuture finished exceptionally
          .exceptionally(notifyCompletionCallbackError(completionCallback));
    } catch (Throwable t) {
      // Set the callback in error if any Throwable exception occurred, include http status code error
      completionCallback.error(t);
    }
  }

}
