package com.mulesoft.connectors.cookBook.internal.connection;

import static com.mulesoft.connectivity.rest.commons.api.error.RestError.CONNECTIVITY;
import static java.util.Optional.of;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.connection.DefaultRestConnection;
import com.mulesoft.connectivity.rest.commons.api.dw.CommonsBindingContext;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectors.cookBook.internal.error.CookBookException;

import java.util.Optional;

public class CookBookSignalErrorHandlingConnection extends DefaultRestConnection {

  public CookBookSignalErrorHandlingConnection(HttpClient httpClient, HttpRequestOptions httpRequestOptions, String baseUri,
                                               ExpressionLanguage expressionLanguage) {
    super(httpClient, httpRequestOptions, baseUri, expressionLanguage);
  }

  @Override
  protected HttpEntity materializeHttpResponseEntity(HttpEntity httpEntity) {
    // Making the current http response repeatable
    return new ByteArrayHttpEntity(IOUtils.toByteArray(httpEntity.getContent()));
  }

  @Override
  protected Optional<ErrorTypeDefinition<RestError>> httpResponseToErrorTypeDefinition(HttpResponse httpResponse, MediaType mediaType) {
    // Status code = 200
    if (httpResponse.getStatusCode() >= 200 && httpResponse.getStatusCode() < 300) {
      // Getting the error property from payload
      Boolean isResponseWithError = getCommonsExpressionLanguage()
              .evaluateAsJavaBoolean("#[payload.error]",
                    CommonsBindingContext.builder()
                        .addJson("payload", httpResponse.getEntity().getContent(),
                                 mediaType.getCharset().orElseThrow(() -> new MuleRuntimeException(
                                     createStaticMessage("mediaType doesn't have a charset definition"))))
                        .build());

      // If error property is true, return connectivity error, at the end, a custom module exception
      // could be returned
      if (isResponseWithError) {
        return of(CONNECTIVITY);
      } else {
        return Optional.empty();
      }
    }
    // Default behavior, it will be catched as a HTTP error
    return super.httpResponseToErrorTypeDefinition(httpResponse, mediaType);
  }

  protected ModuleException createModuleException(HttpResponse httpResponse,
                                                  ErrorTypeDefinition<RestError> errorTypeDefinition,
                                                  MediaType mediaType) {

    // Here a connectivity error was thrown before but the description field is needed in the response
    String description = getCommonsExpressionLanguage()
            .evaluateAsJavaString("#[payload.description]",
                  CommonsBindingContext.builder()
                          .addJson("payload", httpResponse.getEntity().getContent(),
                                   mediaType.getCharset().orElseThrow(() -> new MuleRuntimeException(
                                       createStaticMessage("mediaType doesn't have a charset definition"))))
                      .build());

    return new CookBookException(description, errorTypeDefinition);
  }

}
