package org.mule.extension.maven;

import org.mule.runtime.extension.api.introspection.ExtensionModel;
import org.mule.runtime.extension.api.manifest.ExtensionManifest;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.runtime.extension.api.persistence.manifest.ExtensionManifestXmlSerializer;

import com.google.common.io.Files;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

import org.apache.commons.io.IOUtils;
import org.apache.maven.plugin.MojoFailureException;

/**
 * Utils to share common behaviour
 *
 * @since 1.0
 */
public class ExtensionMojoUtils
{

    private final File outputDirectory;
    private final String artifactName;

    ExtensionMojoUtils(File outputDirectory, String artifactName)
    {
        this.outputDirectory = outputDirectory;
        this.artifactName = artifactName;
    }

    /**
     * @return the {@link ExtensionModel} of the project
     * @throws MojoFailureException if not found
     */
    ExtensionModel getExtensionModel() throws MojoFailureException
    {
        InputStream in = null;
        try (JarFile jar = new JarFile(getExtensionJar()))
        {
            in = getEntryStream(jar, "META-INF/extension-model.json");
            return new ExtensionModelJsonSerializer().deserialize(IOUtils.toString(in));
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Could not obtain [%s] file", e);
        }
        finally
        {
            IOUtils.closeQuietly(in);
        }
    }

    /**
     * @return the {@link ExtensionManifest} of the project
     * @throws MojoFailureException if not found
     */
    ExtensionManifest getExtensionManifest() throws MojoFailureException
    {
        InputStream in = null;
        try (JarFile jar = new JarFile(getExtensionJar()))
        {
            in = getEntryStream(jar, "META-INF/extension-manifest.xml");
            return new ExtensionManifestXmlSerializer().deserialize(IOUtils.toString(in));
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Could not obtain [%s] file", e);
        }
        finally
        {
            IOUtils.closeQuietly(in);
        }
    }

    /**
     * @return the {@link File} pointing to the extension JAR file
     * @throws MojoFailureException if not found
     */
    File getExtensionJar() throws MojoFailureException
    {
        File jar = new File(outputDirectory, artifactName + ".jar");
        if (!jar.exists())
        {
            throw new MojoFailureException(String.format("Extension jar <%1s> doesn't exist in target folder", artifactName + ".jar"));
        }

        return jar;
    }

    /**
     * Given a {@link JarFile} and a entry path, retrieves the correspondent {@link InputStream}
     *
     * @param jar       {@link JarFile} to navigate
     * @param entryPath of the file to obtain their {@link InputStream}
     * @return the {@link InputStream of the required file}
     * @throws MojoFailureException
     * @throws IOException
     */
    private InputStream getEntryStream(JarFile jar, String entryPath) throws MojoFailureException, IOException
    {
        ZipEntry manifestFile = jar.getEntry(entryPath);
        if (manifestFile == null)
        {
            throw new MojoFailureException(String.format("The extension does not contain [%s]", entryPath));
        }

        return jar.getInputStream(manifestFile);
    }

    /**
     * Given a {@link File} directory, creates it if not exist
     *
     * @param directory {@link File} directory to create
     * @throws IOException if not could be created
     */
    public static void createDirectoryIfNotExist(File directory) throws IOException
    {
        if (!directory.exists())
        {
            Files.createParentDirs(directory);
            final boolean createdCorrectly = directory.mkdir();
            if (!createdCorrectly)
            {
                throw new IOException(String.format("The directory [%s] failed to be created", directory));
            }
        }
    }
}
