package org.mule.extension.maven.generator;

import static org.mule.extension.maven.ExtensionMojoUtils.createDirectoryIfNotExist;

import org.mule.api.MuleVersion;
import org.mule.runtime.extension.api.Category;
import org.mule.runtime.extension.api.introspection.ExtensionModel;
import org.mule.runtime.extension.api.manifest.ExtensionManifest;
import org.mule.runtime.extension.xml.dsl.api.property.XmlModelProperty;

import com.google.common.base.CaseFormat;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Optional;

import org.apache.maven.project.MavenProject;

/**
 * This class contains all the information that is required to generate the Studio Extension UpdateSite.
 * Given the {@link MavenProject}, {@link ExtensionManifest} and {@link ExtensionModel} generates all the common
 * information that is shared between generators.
 *
 * @since 1.0
 */
public final class GenerationContext
{

    private static final String STUDIO_PREFIX = "org.mule.tooling.ui.extension.";
    private static final String STUDIO_CONTRIBUTION = "org.mule.tooling.ui.contribution";
    private static final SimpleDateFormat STUDIO_DATE_FORMAT = new SimpleDateFormat("yyyyMMddHHmm");
    private final String artifactName;
    private final String bundleVersion;
    private final String featureId;
    private final String featureJarName;
    private final String extensionPluginName;
    private final String pluginJarName;
    private final String symbolicName;
    private final MavenProject project;
    private final Category extensionCategory;
    private final ExtensionManifest manifest;
    private final ExtensionModel extensionModel;
    private final File outputDirectory;
    private final File classesDirectory;
    private final File pluginOutputDirectory;

    public GenerationContext(MavenProject project, ExtensionManifest manifest, ExtensionModel extensionModel, File outputDirectory) throws IOException
    {
        this.project = project;
        this.manifest = manifest;
        this.extensionModel = extensionModel;

        final String completeMinMuleVersion = extensionModel.getMinMuleVersion().toCompleteNumericVersion();
        this.bundleVersion = buildVersion(project.getVersion());
        this.extensionPluginName = getExtensionNamespace(extensionModel).orElse(CaseFormat.UPPER_CAMEL.to(CaseFormat.LOWER_HYPHEN, manifest.getName().replaceAll(" ", "")));
        this.featureId = STUDIO_PREFIX + removeMuleModulePrefix(project.getArtifactId()) + "." + completeMinMuleVersion;
        this.featureJarName = featureId + "_" + bundleVersion + ".jar";
        this.symbolicName = STUDIO_CONTRIBUTION + "." + extensionPluginName + "." + completeMinMuleVersion;
        this.pluginJarName = symbolicName + "_" + bundleVersion + ".jar";
        this.artifactName = project.getArtifactId() + "-" + project.getVersion();
        this.extensionCategory = extensionModel.getCategory();
        this.outputDirectory = outputDirectory;
        this.classesDirectory = new File(outputDirectory, "classes");
        this.pluginOutputDirectory = new File(classesDirectory, extensionPluginName);
        createDirectoryIfNotExist(pluginOutputDirectory);
    }

    /**
     * @param version to build to the bundle format
     * @return the built plugin version, this is composed by the complete numeric version and a qualifier.
     * <p>
     * <b>Example:</b> {@param version} : '4.0.0', the method returns: '4.0.0.20160630173'
     */
    private static String buildVersion(String version)
    {
        final String completeNumericVersion = new MuleVersion(version).toCompleteNumericVersion();
        final String versionQualifier = STUDIO_DATE_FORMAT.format(new Date());
        return completeNumericVersion + "." + versionQualifier;
    }

    /**
     * @return the namespace of the extension, if not found, the hyphenated name of the extension.
     * <p>
     * <b>Example:</b>
     * <ul>
     * <li><b>Namespace example:</b> 'HTTP Connector' to 'http'</li>
     * <li><b>Hyphenated name example:</b> 'HTTP Connector' to 'http-connector'</li>
     * </ul>
     */
    public String getExtensionPluginName()
    {
        return extensionPluginName;
    }

    /**
     * @return the Feature Id of the extension plugin, this name is composed by the Studio prefix, the artifact Id and
     * the minimum mule version of the extension.
     * <p>
     * <b>Example:</b> 'org.mule.tooling.ui.extension.http.4.0.0'
     */
    public String getFeatureId()
    {
        return featureId;
    }

    /**
     * @return the feature JAR name, this is composed by {@link GenerationContext#getFeatureId()} and the bundle version.
     * <p>
     * <b>Example:</b> 'org.mule.tooling.ui.extension.http.4.0.0_4.0.0.201606291903.jar'
     */
    public String getFeatureJarName()
    {
        return featureJarName;
    }

    /**
     * @return the plugin JAR name, this is composed by the {@link GenerationContext#getSymbolicName()} and the bundle
     * version.
     * <p>
     * <b>Example:</b> 'org.mule.tooling.ui.contribution.http.4.0.0_4.0.0.201606291903.jar'
     */
    public String getPluginJarName()
    {
        return pluginJarName;
    }

    /**
     * @return the extension plugin symbolic name, this is composed by {@link GenerationContext#STUDIO_CONTRIBUTION},
     * the {@link GenerationContext#getExtensionPluginName()} and the minimum mule version of the extension.
     * <p>
     * <b>Example:</b> 'org.mule.tooling.ui.contribution.http.4.0.0'
     */
    public String getSymbolicName()
    {
        return symbolicName;
    }

    /**
     * @return the extension plugin bundle version, this is composed by the version taken of the Maven project and the
     * actual date in 'yyyyMMddHHmm' format.
     * <p>
     * <b>Example:</b> '4.0.0.201606291903'
     */
    public String getBundleVersion()
    {
        return bundleVersion;
    }

    /**
     * @return the extension artifact name, this name is composed by the Maven Artifact Id and the project version.
     * <p>
     * <b>Example:</b> 'mule-module-http-4.0-SNAPSHOT.zip'
     */
    public String getArtifactName()
    {
        return artifactName;
    }

    /**
     * @return the extension's {@link Category}.
     * @see Category
     */
    public Category getExtensionCategory()
    {
        return extensionCategory;
    }

    /**
     * @return the {@link MavenProject} of the extension project
     */
    public MavenProject getMavenProject()
    {
        return project;
    }

    /**
     * @return the extension's {@link ExtensionManifest}
     */
    public ExtensionManifest getExtensionManifest()
    {
        return manifest;
    }

    /**
     * @return the extension's {@link ExtensionModel}
     */
    public ExtensionModel getExtensionModel()
    {
        return extensionModel;
    }

    /**
     * Given an {@param artifactId} checks if has the common 'mule-module-' prefix, and removes it.
     *
     * @param artifactId of the extension
     * @return the processed artifactId
     */
    private String removeMuleModulePrefix(String artifactId)
    {
        String muleModulePrefix = "mule-module-";
        if (artifactId.startsWith(muleModulePrefix))
        {
            return artifactId.substring(muleModulePrefix.length());
        }
        return artifactId;
    }

    /**
     * @return a {@link File} pointing to the plugin output directory
     */
    public File getPluginOutputDirectory()
    {
        return pluginOutputDirectory;
    }

    /**
     * @return a {@link File} pointing to the output/target project directory
     */
    public File getOutputDirectory()
    {
        return outputDirectory;
    }

    /**
     * @return a {@link File} pointing to the output classes directory
     */
    public File getClassesDirectory()
    {
        return classesDirectory;
    }

    /**
     * Given an {@link ExtensionModel} returns the extension XSD Namespace
     *
     * @param extensionModel to retrieve the namespace
     * @return the extension namespace
     */
    private Optional<String> getExtensionNamespace(ExtensionModel extensionModel)
    {
        final Optional<XmlModelProperty> xmlModelProperty = extensionModel.getModelProperty(XmlModelProperty.class);
        return xmlModelProperty.isPresent() ? Optional.of(xmlModelProperty.get().getNamespace()) : Optional.empty();
    }
}
