/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 **/

/**
 * This file was automatically generated by the Mule Development Kit
 */
package org.mule.module.hubspot;

import java.io.IOException;
import java.net.URI;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.PostConstruct;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.JsonParser;
import org.codehaus.jackson.map.ObjectMapper;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.annotations.param.OutboundHeaders;
import org.mule.module.hubspot.exception.HubSpotConnectorAccessTokenExpiredException;
import org.mule.module.hubspot.exception.HubSpotConnectorException;
import org.mule.module.hubspot.exception.HubSpotConnectorNoAccessTokenException;
import org.mule.module.hubspot.model.OAuthCredentials;
import org.springframework.core.annotation.Order;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;

/**
 * HubSpot all-in-one marketing software helps more than 8,000 companies in 56 countries attract leads and convert them into customers. 
 * A pioneer in inbound marketing, HubSpot aims to help its customers make marketing that people actually love.
 * <p>
 * The connector is using the version "v1" of the HubSpot API.
 * <p>
 * The documentation of the API can be found in this <a href="http://developers.hubspot.com/docs">link</a>
 * <p>
 * The main flow of the connector is "authentication" ---> HubSpot Login Page ----> "authenticationResponse" ----> Any other process of the connector
 *
 * @author MuleSoft, Inc.
 */
@Connector(name="hubspot", schemaVersion="1.0", friendlyName="HubSpot")
public class HubSpotConnector
{
	static final private Log logger = LogFactory.getLog(HubSpotConnector.class);
	
	static final private String HUB_SPOT_URL_API 		= "http://hubapi.com";
	static final private String HUB_SPOT_URL_AUTH		= "https://app.hubspot.com/auth/authenticate";
	static final private String PARAM_ACCESS_TOKEN 		= "access_token";
	static final private String API_VERSION				= "v1";
	static final private Pattern PATTERN_ACCESS_TOKEN	= Pattern.compile("access_token=([^&]+)&?");
	static final private Pattern PATTERN_EXPIRES_AT		= Pattern.compile("expires_in=([^&]+)&?");
	static final private Pattern PATTERN_REFRESH_TOKEN	= Pattern.compile("refresh_token=([^&]+)&?");
	static final private Pattern PATTERN_USERID			= Pattern.compile("userid=([^&]+)&?");
	static final private Pattern PATTERN_ERROR			= Pattern.compile("error=([^&]+)&?");
	
	/**
	 * Your Client ID (OAuth Client ID), which identifies who you are. You can access the client_id in your app's developer dashboard under the Summary section.
	 */
	@Configurable
	@Order(1)
	private String clientId;
	
	/**
	 * The HubSpot portal ID of the customer that you're re-directing. You will need to get the portal ID from the customer who you're making the request for.
	 * <p>
	 * In order to find the Hub ID follow this link: <a href="http://help.hubspot.com/articles/How_To_Doc/How-to-find-your-hub-id">http://help.hubspot.com/articles/How_To_Doc/How-to-find-your-hub-id</a>
	 */
	@Configurable
	@Order(2)
	private String hubId;
	
	/**
	 * The scopes (or permissions) you want. These should match your application settings in the Marketplace. Separate more than one scope with "+".	 * 
	 * <p>
	 * <b>Important:</b> the scope offline provides the ability to refresh the token automatically once this has expired. If you do not specify this scope, once the token
	 * expires the call to a process will throw a HubSpotConnectorAccessTokenExpiredException
	 * <p>
	 * <b>Note:</b> the scope required in the authentication must be supported by the application. This can be checked in the Application Settings, under Scopes
	 * <p>
	 * For a complete list of the available scopes check this link: <a href="http://developers.hubspot.com/auth/oauth_scopes">http://developers.hubspot.com/auth/oauth_scopes</a>
	 */
	@Configurable
	@Order(3)
	private String scope;
	
	/**
	 * The callbackUrl is the endpoint that is registered in the iApp to handle the response of the 
	 * authorization call. This endpoint also has to direct to the handleAuthentication process of the connector
	 */
	@Configurable
	@Order(4)
	private String callbackUrl;
	
	private Map<String, OAuthCredentials> credentials;
	
	private Client jerseyClient;
	
	@PostConstruct
	public void initialize() {
		jerseyClient = new Client();
		credentials = new WeakHashMap<String, OAuthCredentials>();
	}

	/**
	 * This process generates the URL required to authenticate against the service.
	 * <p>
	 * <b>Important:</b> in order for the full authentication to work, the callbackUrl in the configuration must be
	 * pointing to another flow that has the authenticateResponse process to handle the reception of the token
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:authenticate}
	 * 
	 * @param userId This user identifier it is the one that will we used from now on to the successive calls to the process of this connector for this user
	 * @param headers This are added implicitly by Studio. The headers of the HTTP inbound, so it can establish a redirect code (302)
	 * @return The URL where the user will be redirected
	 * @throws HubSpotConnectorException If occur some error trying to generate the URL or the userId is empty it will throw this exception.
	 */
	@Processor
	public String authenticate(String userId, @OutboundHeaders Map<String, Object> headers) throws HubSpotConnectorException {
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_AUTH).build();
		WebResource wr = jerseyClient.resource(uri);
		
		String finalCallbackUrl = callbackUrl + (callbackUrl.indexOf('?') < 0 ? "?" : "&") + "userid=" + userId;
				
		wr = wr.queryParam("client_id", clientId)
				.queryParam("portalId", hubId)
				.queryParam("redirect_uri", finalCallbackUrl)
				.queryParam("scope", scope);
		
		String authUrl = wr.getURI().toString();
		
		headers.put("Location", authUrl);
		headers.put("http.status", "302");
		
		logger.info("Ready for authentication. Redirecting (302) to: " + authUrl);
		
		return authUrl;
	}
	
	/**
	 * This process is the one that handles the response of the authentication process. It should be inside an HTTP inbound which
	 * url must be the same that the one pointed by the callbackUrl in the configuration in order to get
	 * the access_token provided by the service.
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:authenticate-response}
	 * 
	 * @param inputRequest The input parameters that came with the response to the authenticate process
	 * @return The UserID that you provided in the call to the authenticate process and that is the one that the user is going to provide in order than the connector use their credentials 
	 * @throws HubSpotConnectorException If any one of the required parameters is empty it will throw this exception.
	 * @throws HubSpotConnectorNoAccessTokenException If there is not an access_token in the response it will throw this exception.
	 */
	@Processor
	public String authenticateResponse(String inputRequest) throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException {
		
		if (StringUtils.isEmpty(inputRequest))
			throw new HubSpotConnectorException("The parameter inputRequest can not be empty");
		
		OAuthCredentials oACreds = new OAuthCredentials();
		
		// Check if the service does not respond with an error
		Matcher m = PATTERN_ERROR.matcher(inputRequest);
		if (m.find()) {
			String errDesc = m.group(1);
			if (errDesc.equals("invalid_scope")) {
				throw new HubSpotConnectorException("The configuration is requesting a scope that the service application does not have available.");
			} else {
				throw new HubSpotConnectorException("The service has responded with an error message: " + errDesc);
			}
		}

		// Save the parameters that appear in the input
		m = PATTERN_USERID.matcher(inputRequest);
		if (m.find()) {
			oACreds.setUserId(m.group(1));
		}

		m = PATTERN_ACCESS_TOKEN.matcher(inputRequest);
		if (m.find()) {
			oACreds.setAccessToken(m.group(1));
		}
		
		m = PATTERN_EXPIRES_AT.matcher(inputRequest);
		if (m.find()) {
			oACreds.setExpiresAt(m.group(1));
		}
		
		m = PATTERN_REFRESH_TOKEN.matcher(inputRequest);
		if (m.find()) {
			oACreds.setRefreshToken(m.group(1));
		}		
		
		// The access token is the only parameter that is absolutely required 
		if (oACreds.getAccessToken() == null) {
			logger.error("Cannot find the access_token in the response:" + inputRequest);
			throw new HubSpotConnectorNoAccessTokenException("The response of the authentication process does not have an access token. Url:" + inputRequest);
		}
		
		credentials.put(oACreds.getUserId(), oACreds);
		logger.info("Stored credentials for user:" + oACreds.getUserId());
		
		return oACreds.getUserId();
	}
	
	/**
	 * Check if the User has an Access Token. This indicate that this User can start calling the process of the connector without any problems
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:has-user-access-token}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @return A boolean that indicates if the user has an access token. Id does not check if the token is or not expired
	 */
	@Processor
	public boolean hasUserAccessToken(String userId) {
		return credentials.get(userId) != null && StringUtils.isNotEmpty(credentials.get(userId).getAccessToken());
	}	
	
	//
	/**
	 * For a given portal, return all contacts that have been created in the portal.
	 * A paginated list of contacts will be returned to you, with a maximum of 100 contacts per page.
	 * <p>
	 * API Link: <a href="http://developers.hubspot.com/docs/methods/contacts/get_contacts">http://developers.hubspot.com/docs/methods/contacts/get_contacts</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-all-contacts}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param count This parameter lets you specify the amount of contacts to return in your API call. The default for this parameter (if it isn't specified) is 20 contacts. The maximum amount of contacts you can have returned to you via this parameter is 100.
	 * @param contactOffset This parameter will offset the contacts returned to you, based on the unique ID of the contacts in a given portal. Contact unique IDs are assigned by the order that they are created in the system. This means for instance, if you specify a vidOffset offset of 5, and you have 20 contacts in the portal you're working in, the contacts with IDs 6-20 will be returned to you.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getAllContacts(String userId, @Optional @Default("") String count, @Optional @Default("") String contactOffset) 
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/lists/all/contacts/all").build(API_VERSION);

		WebResource wr = getWebResource(userId, uri);				
		if (count != null) wr = wr.queryParam("count", count);		
		if (contactOffset != null) wr = wr.queryParam("vidOffset", contactOffset);
		
		logger.info("Requesting allContacts to: " + wr.toString());		
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
				
		return strResponse;
	}
	
	
	/**
	 * For a given portal, return all contacts that have been recently updated or created.
	 * A paginated list of contacts will be returned to you, with a maximum of 100 contacts per page, as specified by the "count" parameter.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/get_recently_updated_contacts">http://developers.hubspot.com/docs/methods/contacts/get_recently_updated_contacts</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-recent-contacts}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param count This parameter lets you specify the amount of contacts to return in your API call. The default for this parameter (if it isn't specified) is 20 contacts. The maximum amount of contacts you can have returned to you via this parameter is 100.
	 * @param timeOffset Used in conjunction with the vidOffset paramter to page through the recent contacts. Every call to this endpoint will return a time-offset value. This value is used in the timeOffset parameter of the next call to get the next page of contacts.
	 * @param contactOffset Used in conjunction with the timeOffset paramter to page through the recent contacts. Every call to this endpoint will return a vid-offset value. This value is used in the vidOffset parameter of the next call to get the next page of contacts.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getRecentContacts(String userId, @Optional @Default("") String count, @Optional @Default("") String timeOffset, @Optional @Default("") String contactOffset)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/lists/recently_updated/contacts/recent").build(API_VERSION);
		
		WebResource wr = getWebResource(userId, uri);
		if (count != null) 			wr = wr.queryParam("count", count);
		if (timeOffset != null) 	wr = wr.queryParam("timeOffset", timeOffset);
		if (contactOffset != null) 	wr = wr.queryParam("vidOffset", contactOffset);
		
		logger.info("Requesting recentContacts to:" + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
				
		return strResponse;
	}
	
	
	/**
	 * For a given portal, return information about a single contact by its ID. The contact's unique ID's is stored in a field called 'vid' which stands for 'visitor ID'.
	 * This method will also return you much of the HubSpot lead "intelligence" that you may be accustomed to getting from the leads API, as properties in this new API. 
	 * More of this intelligence will be available as time passes, but this call is where you can expect to find it.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/get_contact">http://developers.hubspot.com/docs/methods/contacts/get_contact</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contact-by-id}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param contactId Unique identifier for a particular contact. In HubSpot's contact system, contact ID's are called "vid".
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactById(String userId, String contactId)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(contactId))
			throw new HubSpotConnectorException("The parameter contactId cannot be empty");
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contact/vid/{contactid}/profile").build(API_VERSION, contactId);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting contactById to:" + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
		
		return strResponse;
	}
	
	/**
	 * For a given portal, return information about a single contact by its email address.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/get_contact_by_email">http://developers.hubspot.com/docs/methods/contacts/get_contact_by_email</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contact-by-email}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param contactEmail The email address for the contact that you're searching for.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactByEmail(String userId, String contactEmail)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(contactEmail))
			throw new HubSpotConnectorException("The parameter contactEmail cannot be empty");
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contact/email/{contactemail}/profile").build(API_VERSION, contactEmail);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting contactByEmail to:" + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET); 
				
		return strResponse;
	}
	
	/**
	 * For a given portal, return information about a single contact by its User Token (hubspotutk)
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/get_contact_by_utk">http://developers.hubspot.com/docs/methods/contacts/get_contact_by_utk</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contact-by-user-token}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param contactUserToken The user token (HubSpot cookie) for the contact that you're searching for.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactByUserToken(String userId, String contactUserToken)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(contactUserToken))
			throw new HubSpotConnectorException("The parameter contactUserToken cannot be empty");
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contact/utk/{contactusertoken}/profile").build(API_VERSION, contactUserToken);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting contactByUserToken to: " +  wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
		
		return strResponse;
	}
	
	/**
	 * For a given portal, return contacts and some data associated with those contacts by the contact's email address or name.
	 * Please note that you should expect this method to only return a small subset of data about the contact. One piece of data 
	 * that the method will return is the contact ID (vid) that you can then use to look up much more data about that particular contact by its ID.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/search_contacts">http://developers.hubspot.com/docs/methods/contacts/search_contacts</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contacts-by-query}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param query The search term for what you're searching for. You can use all of a word or just parts of a word as well. For example, if you we're searching for contacts with "hubspot" in their name or email, searching for "hub" would also return contacts with "hubspot" in their email address.
	 * @param count This parameter lets you specify the amount of contacts to return in your API call. The default for this parameter (if it isn't specified) is 20 contacts. The maximum amount of contacts you can have returned to you via this parameter is 100.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactsByQuery(String userId, String query, @Optional @Default("") String count)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(query))
			throw new HubSpotConnectorException("The parameter query cannot be empty");
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/search/query").build(API_VERSION);
		
		WebResource wr = getWebResource(userId, uri);
		wr = wr.queryParam("q", query);
		if (count != null) wr = wr.queryParam("count", count);
		
		logger.info("Requesting contactsByQuery to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
		
		return strResponse;
	}
		
	/**
	 * Archive an existing contact from a particular HubSpot portal. 
	 * Archiving will not hard delete a contact from a portal, but will remove that contact from the HubSpot user interface.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/delete_contact">http://developers.hubspot.com/docs/methods/contacts/delete_contact</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:delete-contact}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param contactId You must pass the Contact's ID that you're archiving in the request URL.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String deleteContact(String userId, String contactId)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(contactId))
			throw new HubSpotConnectorException("The parameter contactId cannot be empty");
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contact/vid/{contactid}").build(API_VERSION, contactId);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting deleteContact to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.DELETE);
		
		return strResponse;
		
	}
	
	/**
	 * Update an existing contact in HubSpot. This method lets you update one of many fields of a contact in HubSpot.
	 * <p>
	 * To update a contact, you should make an HTTP POST call to this endpoint with some JSON in the request payload. 
	 * This JSON should contain properties from the contact that you want to add to or update. See the sample JSON below for an example of this snippet of JSON.
	 * <p>
	 * If you are trying to close a contact into a customer via the API, you should be updating the 'lifecyclestage' property and setting the value of this property to 'customer'.
	 * <p>
	 * Remember, if a property doesn't yet exist, you can create a new custom property through the API by using the 'Create Property' method.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/update_contact">http://developers.hubspot.com/docs/methods/contacts/update_contact</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:update-contact}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param contactId You must pass the Contact's ID that you're updating in the request URL
	 * @param contactJson This is JSON that represents a contact that you're updating. <b>Important:</b> use " and not ' in your JSON.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String updateContact(String userId, String contactId, String contactJson)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(contactId))
			throw new HubSpotConnectorException("The parameter contactId cannot be empty");
		if (StringUtils.isEmpty(contactJson))
			throw new HubSpotConnectorException("The parameter contactJson cannot be empty");
		checkValidJson(contactJson);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contact/vid/{contactid}/profile").build(API_VERSION, contactId);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting updateContact to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.POST, contactJson);
				
		return strResponse;
	}
	
	/**
	 * Create a new contact in HubSpot with a simple HTTP POST to the Contacts API.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/create_contact">http://developers.hubspot.com/docs/methods/contacts/create_contact</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:create-contact}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param contactJson This is JSON that represents a contact that you're creating. <b>Important:</b> use " and not ' in your JSON. <b>Note:</b> The property email is mandatory.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String createContact(String userId, String contactJson)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		if (StringUtils.isEmpty(contactJson))
			throw new HubSpotConnectorException("The parameter contactJson cannot be empty");
		checkValidJson(contactJson);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contact").build(API_VERSION);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting createContact to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.POST, contactJson);
		
		return strResponse;
	}
	
	/**
	 * For a given portal, return statistics about that portal's contacts.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/contacts/get_contact_statistics">http://developers.hubspot.com/docs/methods/contacts/get_contact_statistics</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contact-statistics}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactStatistics(String userId)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/contacts/statistics").build(API_VERSION);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting contactStatistics to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
		
		return strResponse;
	}
	
	/**
	 * For a given portal, return a set of contact lists that you specify with the count parameter.
	 * By default, we will only return up to 20 lists to you at a time.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/lists/get_lists">http://developers.hubspot.com/docs/methods/lists/get_lists</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contacts-lists}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param count An integer that represents the number of lists that you want returned to your call. By default, this call will return 20 lists to you. If you want more or different list returned to you, you'll want to use the "offset" parameter.
	 * @param offset An integer that represents where to start your list pull from. For instance, if you want to return numbered lists: 50-60, your offset should be "50" and your count parameter (seen above) should be 10. You should also note that the returned JSON (seen below) includes a "has-more" field, which lets you know if there are more lists that you can pull. If "has-more" is true, you can use this offset parameter to pull lists that weren't in your initial call.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactsLists(String userId, @Optional @Default("") String count, @Optional @Default("") String offset) 
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/lists").build(API_VERSION);
		
		WebResource wr = getWebResource(userId, uri);
		if (count != null) wr = wr.queryParam("count", count);
		if (offset != null) wr = wr.queryParam("offset", offset);
		
		logger.info("Requesting contactsLists to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
				
		return strResponse;
	}
	
	/**
	 * For a given portal, return a contact list by its unique ID.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/lists/get_list">http://developers.hubspot.com/docs/methods/lists/get_list</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-contact-list-by-id}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param listId Unique identifier for the list that you're looking for.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getContactListById(String userId, String listId)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/lists/{listid}").build(API_VERSION, listId);
		
		WebResource wr = getWebResource(userId, uri);
		
		logger.info("Requesting contactListById to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
		
		return strResponse;
	}
	
	/**
	 * For a given portal, return a set of dynamic contact lists that you specify with the count parameter.
	 * <p>
	 * Dynamic lists are lists that can only be edited by the contacts app - they are meant to update themselves 
	 * when new contacts are created or are updated, meaning that you can't manually add contacts to dynamic lists.
	 * <p>
	 * By default, we will only return 20 lists to you via this API call.
	 * <p>
	 * API link: <a href="http://developers.hubspot.com/docs/methods/lists/get_dynamic_lists">http://developers.hubspot.com/docs/methods/lists/get_dynamic_lists</a>
	 * <p>
	 * {@sample.xml ../../../doc/HubSpot-connector.xml.sample hubspot:get-dynamic-contact-lists}
	 * 
	 * @param userId The UserID of the user in the HubSpot service that was obtained from the {@link authenticateResponse} process
	 * @param count An integer that represents the number of lists that you want returned to your call. By default, this call will return 20 lists to you. If you want more or different list returned to you, you'll want to use the "offset" parameter.
	 * @param offset An integer that represents where to start your list pull from. For instance, if you want to return numbered lists: 50-60, your offset should be "50" and your count parameter (seen above) should be 10. You should also note that the returned JSON (seen below) includes a "has-more" field, which lets you know if there are more lists that you can pull. If "has-more" is true, you can use this offset parameter to pull lists that weren't in your initial call.
	 * @return A JSON format response from the service
	 * @throws HubSpotConnectorException If the required parameters were not specified or occurs another type of error this exception will be thrown
	 * @throws HubSpotConnectorNoAccessTokenException If the user does not have an Access Token this exception will be thrown
	 * @throws HubSpotConnectorAccessTokenExpiredException If the user has his token already expired this exception will be thrown
	 */
	@Processor
	public String getDynamicContactLists(String userId, @Optional @Default("") String count, @Optional @Default("") String offset)
			throws HubSpotConnectorException, HubSpotConnectorNoAccessTokenException, HubSpotConnectorAccessTokenExpiredException {
		
		checkEmptyUserId(userId);
		
		URI uri = UriBuilder.fromPath(HUB_SPOT_URL_API).path("/contacts/{apiversion}/lists/dynamic").build(API_VERSION);
		
		WebResource wr = getWebResource(userId, uri);
		if (count != null) wr = wr.queryParam("count", count);
		if (offset != null) wr = wr.queryParam("offset", offset);
		
		logger.info("Requesting dynamicContactLists to: " + wr.toString());
		String strResponse = webResourceGet(wr, userId, WebResourceMethods.GET);
		
		return strResponse;
	}
	
	/**
	 * Method used to eliminate boilerplate handling exceptions when calling get(String.class) from a WebResource
	 * 
	 * @param wr The WebResource to call get
	 * @param userId The userId from the session used
	 * @return The response of the service in String format
	 * @throws HubSpotConnectorAccessTokenExpiredException If the service responded with a 401 means that the session has expired
	 * @throws HubSpotConnectorException If is not a 401 it will throw this exception
	 */
	private String webResourceGet(WebResource wr, String userId, WebResourceMethods method) 
			throws HubSpotConnectorAccessTokenExpiredException, HubSpotConnectorException {
		return webResourceGet(wr, userId, method, null);
	}
	
	
	private String webResourceGet(WebResource wr, String userId, WebResourceMethods method, String requestBody) 
			throws HubSpotConnectorAccessTokenExpiredException, HubSpotConnectorException {
		try {
			return webResourceCallByEnumType(wr, method, requestBody);
		} catch (UniformInterfaceException e) {
			int statusCode = e.getResponse().getStatus();
			
			// The code 204 is returned as a successful operation with no response, but as the expected parameter is a String.class it throws a UniformInterfaceException.
			if (statusCode == 204) {
				return "";
			} else if (statusCode == 401) {
				// TODO: Add verification if it has a refresh token. If it has it call refresh process and then make again the get() call
				throw new HubSpotConnectorAccessTokenExpiredException("The access token for the userId " + userId + "has expired", e);
			} else {
				throw new HubSpotConnectorException("ERROR - statusCode: " + statusCode, e);
			}
		}
	}
	
	private String webResourceCallByEnumType(WebResource wr, WebResourceMethods method, String requestBody) {
		if (WebResourceMethods.GET.equals(method)) {
			return wr.get(String.class);
		} else if (WebResourceMethods.POST.equals(method)) {
			return wr.type(MediaType.APPLICATION_JSON_TYPE).post(String.class, requestBody);
		} else if (WebResourceMethods.PUT.equals(method)) {
			return wr.put(String.class);
		} else if (WebResourceMethods.DELETE.equals(method)) {
			return wr.delete(String.class);
		} else {
			return null;
		}
	}
	
	static private enum WebResourceMethods {
		GET,
		POST,
		PUT,
		DELETE;
	}
	
	private void checkValidJson(final String json) throws HubSpotConnectorException {
	   try {
	      final JsonParser parser = new ObjectMapper().getJsonFactory().createJsonParser(json);
	      while (parser.nextToken() != null) {}
	   } catch (JsonParseException jpe) {
		   throw new HubSpotConnectorException("The contactJson contains a JSON malformed", jpe);
	   } catch (IOException ioe) {
		   throw new HubSpotConnectorException("The contactJson cannot be readed", ioe);
	   }
	}
	
	private WebResource getWebResource(String userId, URI uri) throws HubSpotConnectorNoAccessTokenException {
		OAuthCredentials oACreds = credentials.get(userId);
		
		if (oACreds == null)
			throw new HubSpotConnectorNoAccessTokenException("The user with id " + userId + " does not have credentials");
		
		return jerseyClient.resource(uri).queryParam(PARAM_ACCESS_TOKEN, oACreds.getAccessToken());
	}
	
	private void checkEmptyUserId(String userId) throws HubSpotConnectorException {
		if (StringUtils.isEmpty(userId))
			throw new HubSpotConnectorException("The parameter UserId cannot be empty");
	}

	public String getClientId() {
		return clientId;
	}

	public void setClientId(String clientId) {
		this.clientId = clientId;
	}

	public String getHubId() {
		return hubId;
	}

	public void setHubId(String hubId) {
		this.hubId = hubId;
	}

	public String getScope() {
		return scope;
	}

	public void setScope(String scope) {
		this.scope = scope;
	}

	public String getCallbackUrl() {
		return callbackUrl;
	}

	public void setCallbackUrl(String callbackUrl) {
		this.callbackUrl = callbackUrl;
	}

	public Client getJerseyClient() {
		return jerseyClient;
	}

	public void setJerseyClient(Client jerseyClient) {
		this.jerseyClient = jerseyClient;
	}
}
