/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extensions.java.internal.cache;

import static org.mockito.Mockito.*;
import static java.util.Collections.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mule.extensions.java.api.exception.NoSuchConstructorModuleException;
import org.mule.extensions.java.internal.parameters.ConstructorIdentifier;
import org.mule.extensions.java.internal.parameters.ExecutableIdentifier;
import org.mule.runtime.api.metadata.TypedValue;

import java.lang.reflect.Executable;
import java.lang.reflect.Method;
import java.util.*;

public class JavaModuleLoadingCacheTest {

  private JavaModuleLoadingCache cache;
  private Class<?> declaringClass;
  private Map<String, TypedValue<Object>> args;

  @Rule
  public ExpectedException thrown = ExpectedException.none();

  @Before
  public void setup() {
    cache = new JavaModuleLoadingCache();
    declaringClass = String.class;
    args = new HashMap<>();
  }

  @Test
  public void testThrowsNoSuchConstructorModuleExceptionWhenEmptyAndConstructorId() {
    ConstructorIdentifier id = mock(ConstructorIdentifier.class);

    thrown.expect(NoSuchConstructorModuleException.class);
    cache.verifyExecutables(id, declaringClass, args, Collections.emptyList());
  }

  @Test
  public void testDoesNotThrowWhenExactlyOneExecutable() {
    ExecutableIdentifier id = mock(ExecutableIdentifier.class);
    Executable executable = mock(Executable.class);
    List<Executable> executables = singletonList(executable);

    // Should not throw
    cache.verifyExecutables(id, declaringClass, args, executables);
  }

  @Test
  public void testLogsWarningWhenMultipleExecutables() {
    class Dummy {

      public void myMethod() {}

      public void myMethod(String arg) {}
    }

    Method method1 = null;
    Method method2 = null;
    try {
      method1 = Dummy.class.getMethod("myMethod"); // no-arg
      method2 = Dummy.class.getMethod("myMethod", String.class); // one-arg
    } catch (NoSuchMethodException e) {
      fail("Failed to get methods via reflection");
    }

    List<Executable> executables = Arrays.asList(method1, method2);

    // Mock ExecutableIdentifier
    ExecutableIdentifier id = mock(ExecutableIdentifier.class);
    when(id.getElementId()).thenReturn("myMethod");
    when(id.getExecutableTypeName()).thenReturn("method");

    // Should not throw
    cache.verifyExecutables(id, Dummy.class, new HashMap<>(), executables);
  }

}
