package org.mule.module.xmlsecurity.reference;

import org.mule.module.xmlsecurity.algorithms.DigestMethodAlgorithm;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.crypto.XMLStructure;
import javax.xml.crypto.dom.DOMStructure;
import javax.xml.crypto.dsig.Reference;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static org.mule.module.xmlsecurity.Signer.fac;

/**
 * <p>Generates a Reference to an object in the same xml document</p>
 *
 * @author fernando.federico@mulesource.com
 */
public class EnvelopingReferenceProvider implements ReferenceProvider {
    
    private DigestMethodAlgorithm digestMethodAlgorithm;

    /**
     * Constructor
     *
     * @param digestMethodAlgorithm The digest method algorithm
     */
    public EnvelopingReferenceProvider(DigestMethodAlgorithm digestMethodAlgorithm) {
        this.digestMethodAlgorithm = digestMethodAlgorithm;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Reference getReference() {
        try{
            return fac.newReference("#AuthenticationObject",
                    fac.newDigestMethod(digestMethodAlgorithm.getAlgorithm(), null));
        }
        catch(Exception e)
        {
            throw new RuntimeException("Could not getReference Enveloping Reference", e);
        }
        
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Object> getReferencedObjects() {

        List<Object> objects = new ArrayList<Object>();
        try {
            DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
            Document document = builder.newDocument();
            Element authentication = document.createElement("authentication");
            Element nonce = document.createElement("nonce");
            nonce.setTextContent(generateNonce());
            authentication.appendChild(nonce);

            Element timestamp = document.createElement("timestamp");
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            timestamp.setTextContent(sdf.format(new Date().getTime()-(60*60*1000))  );

            authentication.appendChild(timestamp);

            XMLStructure structure = new DOMStructure(authentication);

            objects.add(fac.newXMLObject(Collections.singletonList(structure), "AuthenticationObject",null, "UTF-8"));
            return objects;
        } catch (ParserConfigurationException e) {
            throw new RuntimeException("Error creating objects", e);
        }

    }

    private String generateNonce()
    {
        // Create a secure random number generator

        return key(32)+"-"+key(16)+"-"+key(16)+"-"+key(16)+"-"+key(64);
    }

    private String key(int f) {
        return new BigInteger(f, new SecureRandom()).toString(32);
    }
}
